# -*- coding: utf-8 -*-

from __future__ import unicode_literals

from oauthlib.oauth1 import (
    Client as Oauth1Client,
    SIGNATURE_TYPE_AUTH_HEADER as OAUTH1_SIGNATURE_TYPE_AUTH_HEADER,
    SIGNATURE_TYPE_QUERY as OAUTH1_SIGNATURE_TYPE_QUERY,
)
from passport.backend.social.common.misc import urlparse_qsl
from passport.backend.social.common.useragent import (
    Request,
    Url,
)


class Oauth1RequestSigner(object):
    SIGNATURE_TYPE = OAUTH1_SIGNATURE_TYPE_AUTH_HEADER

    def __init__(self, client_id, client_secret, signature_type=None,
                 oauth_callback=None, oauth_token=None, oauth_token_secret=None,
                 verifier=None, oauth_nonce=None, oauth_timestamp=None):
        self._client_id = client_id
        self._client_secret = client_secret
        self._signature_type = signature_type or self.SIGNATURE_TYPE
        self._oauth_callback = oauth_callback
        self._oauth_token = oauth_token
        self._oauth_token_secret = oauth_token_secret
        self._verifier = verifier
        self._oauth_nonce = oauth_nonce
        self._oauth_timestamp = oauth_timestamp

    def get_signed_request(self, request):
        client = Oauth1Client(
            client_key=self._client_id,
            client_secret=self._client_secret,
            signature_type=self._signature_type,
            callback_uri=self._oauth_callback,
            resource_owner_key=self._oauth_token,
            resource_owner_secret=self._oauth_token_secret,
            verifier=self._verifier,
            nonce=self._oauth_nonce,
            timestamp=self._oauth_timestamp,
        )

        # Чтобы в URL не оказалось два Oauth1 параметра с разным значением,
        # очистим его от всех Oauth1 параметров.
        # Это может пригодится в тех случаях, когда Oauth1 параметры передаются
        # в query, а HTTP-сервер отвечает редиректом, указывая в location
        # другой URL, но со старыми значениями параметров.
        url = self._get_url_without_oauth1_params(request)

        if request.method in {'POST', 'PUT', 'PATCH'}:
            data = urlparse_qsl(request.data or '')
        else:
            data = None

        headers = request.headers
        if not data and 'Content-Type' in headers:
            # sign не умеет работать когда нет data, но есть Content-Type
            headers = dict(headers)
            del headers['Content-Type']

        url, headers, data = client.sign(
            uri=url,
            http_method=request.method,
            # sign не умеет работать с пустым body
            body=data or None,
            headers=headers,
        )

        return Request.from_request(
            request,
            url=url,
            headers=headers,
            data=data,
        )

    def _get_url_without_oauth1_params(self, request):
        OAUTH1_PARAMS = {
            'oauth_body_hash',
            'oauth_callback',
            'oauth_consumer_key',
            'oauth_nonce',
            'oauth_signature',
            'oauth_signature_method',
            'oauth_timestamp',
            'oauth_token',
            'oauth_verifier',
            'oauth_version',
        }
        url = Url(request.url)
        url.remove_params(OAUTH1_PARAMS)
        return str(url)


__all__ = ['OAUTH1_SIGNATURE_TYPE_QUERY']
