# -*- coding: utf-8 -*-

from __future__ import unicode_literals

from .base import (
    BaseInvalidClient,
    BaseInvalidGrant,
    BaseInvalidRequest,
    BaseInvalidScope,
    BaseUnauthorizedClient,
    BaseUnsupportedGrantType,
    UnexpectedException,
)


class InvalidRequest(BaseInvalidRequest):
    """
    The request is missing a required parameter, includes an
    unsupported parameter value (other than grant type),
    repeats a parameter, includes multiple credentials,
    utilizes more than one mechanism for authenticating the
    client, or is otherwise malformed.
    """


class InvalidClient(BaseInvalidClient):
    """
    Client authentication failed (e.g., unknown client, no
    client authentication included, or unsupported
    authentication method).  The authorization server MAY
    return an HTTP 401 (Unauthorized) status code to indicate
    which HTTP authentication schemes are supported.  If the
    client attempted to authenticate via the "Authorization"
    request header field, the authorization server MUST
    respond with an HTTP 401 (Unauthorized) status code and
    include the "WWW-Authenticate" response header field
    matching the authentication scheme used by the client.
    """


class InvalidGrant(BaseInvalidGrant):
    """
    The provided authorization grant (e.g., authorization
    code, resource owner credentials) or refresh token is
    invalid, expired, revoked, does not match the redirection
    URI used in the authorization request, or was issued to
    another client.
    """


class UnauthorizedClient(BaseUnauthorizedClient):
    """
    The authenticated client is not authorized to use this
    authorization grant type.
    """


class UnsupportedGrantType(BaseUnsupportedGrantType):
    """
    The authorization grant type is not supported by the
    authorization server.
    """


class InvalidScope(BaseInvalidScope):
    """
    The requested scope is invalid, unknown, malformed, or
    exceeds the scope granted by the resource owner.
    """


def detect_error(response):
    if not isinstance(response, dict):
        # Если ответ -- это не словарь с ключом error, считаем что отказа не
        # обнаружено. Дальнейшее обнаружение отказа должен взять на себя парсер
        # ответа.
        return

    error_code = response.get('error')
    if error_code or error_code == 0:
        try:
            exception_class = _error_code_to_exception.get(error_code, UnexpectedException)
        except TypeError:
            # Error_code может оказаться не строкой, а например списком
            raise UnexpectedException()

        exception = exception_class()
        exception.error_description = response.get('error_description')
        exception.error_uri = response.get('error_uri')

        if exception.error is None:
            exception.error = error_code
        raise exception


_error_code_to_exception = dict([(cls.error, cls) for cls in [
    InvalidRequest,
    InvalidClient,
    InvalidGrant,
    UnauthorizedClient,
    UnsupportedGrantType,
    InvalidScope,
]])
