# -*- coding: utf-8 -*-

from collections import namedtuple
from datetime import datetime
import logging

from passport.backend.core.db.utils import insert_with_on_duplicate_key_update
from passport.backend.core.utils.blackbox import get_many_accounts_by_uids
from passport.backend.social.common.db.execute import execute
from passport.backend.social.common.db.schemas import (
    business_application_map_table as bamt,
    person_table as pert,
    profile_table as pt,
    sub_table as st,
    token_table as tt,
)
from passport.backend.social.common.db.utils import get_slave_engine
from passport.backend.social.common.limits import get_qlimits
from passport.backend.social.common.misc import (
    get_business_userid,
    parse_userid,
    USERID_TYPE_BUSINESS,
)
from passport.backend.social.common.provider_settings import providers
from passport.backend.social.common.providers.Facebook import Facebook
from passport.backend.social.common.providers.Kinopoisk import Kinopoisk
from passport.backend.social.common.providers.Yandex import Yandex
from passport.backend.social.common.refresh_token.utils import save_refresh_token
from passport.backend.social.common.serialize import (
    serialize_date,
    serialize_unixtime,
)
from passport.backend.social.common.token.utils import (
    find_token_by_value_for_account,
    save_token,
)
from sqlalchemy import (
    and_,
    or_,
)
from sqlalchemy.sql.expression import (
    delete,
    select,
    update,
)


logger = logging.getLogger(__name__)


class BaseProfileCreationError(Exception):
    description = None


class _ProviderUnknownProfileCreationError(BaseProfileCreationError):
    description = 'Unknown provider'


class _ProviderUseridUnknownProfileCreationError(BaseProfileCreationError):
    description = 'Unknown provider user id'


class _ApplicationUnknownProfileCreationError(BaseProfileCreationError):
    description = 'Unknown application'


class AccountNotFoundProfileCreationError(BaseProfileCreationError):
    description = 'Account not found'


class DatabaseFailedProfileCreatetionError(BaseProfileCreationError):
    description = 'Database failed'


class BaseProfileNotAllowedProfileCreationError(BaseProfileCreationError):
    pass


class _MasterMatchesSlaveProfileCreationError(BaseProfileNotAllowedProfileCreationError):
    description = 'Master account matches slave account'


class _MasterAccountTypeInvalidProfileCreationError(BaseProfileNotAllowedProfileCreationError):
    description = 'Invalid master account type'


class _SlaveAccountTypeInvalidCreationError(BaseProfileNotAllowedProfileCreationError):
    description = 'Invalid slave account type'


def get_profile(mysql_read, task, uid):
    """
    Вернем профиль из базы.
    Если профилей в базе два, то вернем первый попавшийся. Этот случай считаем крайне редким,
    поэтому не обрабатываем как-то особо.
    """
    profile_query = get_profile_getting_query_by_profile_info(
        task.get_social_userinfo(),
        uid,
    )
    return execute(mysql_read, select([pt]).where(profile_query)).fetchone()


def find_profiles_by_userid(userid, provider_id, force_all=False):
    query = (
        select([pt])
        .where(
            and_(
                pt.c.provider_id == provider_id,
                pt.c.userid == userid,
            ),
        )
        .order_by(pt.c.profile_id)
    )
    if not force_all:
        query = query.limit(get_qlimits()['profiles'])
    return execute(get_slave_engine(), query).fetchall()


def update_business_userid_map(mysql_read, mysql_write, profile_id, business_userid, simple_userid, userid_in_db, app_id):
    """
    Обновляем данные в таблице `business_application_map`.
    Функция вызывается после обновления данных на профиле или после создания нового профиля.
    profile_id - идентификатор профиля в базе.
    business_userid - Составной userid, который нужно записать в базу.
    simple_userid - userid в соц сети.
    userid_in_db - Такой userid сейчас записан у профиля в базе.
    """

    _, (business_id, business_token) = parse_userid(business_userid)
    values = {
        'business_id': business_id,
        'business_token': business_token,
        'application_id': app_id,
        'userid': simple_userid,
    }

    if userid_in_db != business_userid:
        # Обновляем userid и добавляем соответствие в базу.

        # Сделаем запись для метаприложения 0, чтобы запомнить,
        # каким был userid до перехода на бизнес-userid.
        # Это потенциально нужно для обратной совместимости, удалить всегда успеем.
        values_meta_app = dict(values, application_id=0)

        for vals in [values_meta_app, values]:
            execute(
                mysql_write,
                insert_with_on_duplicate_key_update(bamt, ['business_token']).values(**vals),
            )

        # TODO после обновления sqlalchemy заменить на один запрос:
        # execute(mysql_write, insert(
        #     bamt,
        #     on_duplicate_update_keys={'business_token': business_token},  # игнорируем ошибки дублирования
        # ).values([values_meta_app, values]))

        execute(
            mysql_write,
            update(pt).values(userid=business_userid).where(pt.c.profile_id == profile_id),
        )
    else:
        # userid на профиле уже правильный, его трогать не надо.
        # Однако, записи в business_application_map для текущего приложения может и не быть.

        record = execute(
            mysql_read,
            select([bamt]).where(
                (bamt.c.business_id == business_id) &
                (bamt.c.business_token == business_token) &
                (bamt.c.application_id == app_id)
            ),
        ).fetchone()

        if record:
            # Запись уже есть, больше ничего делать не нужно.
            return
        else:
            # Нужно добавить запись.
            execute(
                mysql_write,
                insert_with_on_duplicate_key_update(bamt, ['business_token']).values(**values),
            )


def _merge_conflicting_profiles(mysql_write, business_profile, simple_profile):
    """
    Удалим старый профиль с простым userid, перенесем какие-то данные с него на новый.
    Вернем оставшийся профиль.
    """
    logger.info(
        'Conflicting profiles found! Simple profile: %s; Business profile: %s',
        ','.join([str(simple_profile.profile_id), simple_profile.userid]),
        ','.join([str(business_profile.profile_id), business_profile.userid])
    )
    # Перенесем токены. Здесь не может быть конфликтов.
    execute(
        mysql_write,
        update(tt).values(
            profile_id=business_profile.profile_id,
        ).where(
            tt.c.profile_id == simple_profile.profile_id,
        ),
    )

    # Не оставим пользователя без средства входа в аккаунт:
    if simple_profile.allow_auth and not business_profile.allow_auth:
        execute(
            mysql_write,
            update(pt).values(
                allow_auth=1,
            ).where(
                pt.c.profile_id == business_profile.profile_id,
            ),
        )

    # Данные из person не обновляем, они и так обновятся дальше.
    # Подписки не переносим, потому что не знаем, какие из них более актуальны.

    # И наконец удалим старый профиль.
    execute(
        mysql_write,
        delete(pt).where(
            pt.c.profile_id == simple_profile.profile_id,
        )
    )

    return business_profile


def get_business_userid_from_profile_info(profile_info):
    provider_info = providers.get_provider_info_by_name(profile_info['provider']['code'])

    if provider_info['id'] == Facebook.id and 'business' in profile_info:
        business_id = profile_info['business']['id']
        business_token = profile_info['business']['token']
        if business_token and business_id:
            return get_business_userid(business_id, business_token)


def get_profile_getting_query_by_profile_info(profile_info, uid):
    """
    Сгенерируем запрос, которым можно получить из базы профиль пользователя
    с учетом потенциального business_userid.
    """
    provider_info = providers.get_provider_info_by_name(profile_info['provider']['code'])
    business_userid = get_business_userid_from_profile_info(profile_info)

    if business_userid:
        return (
            (pt.c.provider_id == provider_info['id']) &
            or_(pt.c.userid == profile_info['userid'], pt.c.userid == business_userid) &
            (pt.c.uid == uid)
        )
    else:
        return (
            (pt.c.provider_id == provider_info['id']) &
            (pt.c.userid == profile_info['userid']) &
            (pt.c.uid == uid)
        )


def update_many_profiles(profile_ids, attributes, write_conn):
    profile_values = dict()
    for col in pt.columns:
        if col.name in attributes:
            profile_values[col.name] = attributes[col.name]
    query = (
        pt.update()
        .values(**profile_values)
        .where(pt.c.profile_id.in_(profile_ids))
    )
    execute(write_conn, query)

    person_values = dict()
    for col in pert.columns:
        if col.name in attributes:
            person_values[col.name] = attributes[col.name]

    if 'birthday' in person_values:
        person_values['birthday'] = serialize_date(person_values['birthday'])

    query = (
        pert.update()
        .values(**person_values)
        .where(pert.c.profile_id.in_(profile_ids))
    )
    execute(write_conn, query)


def create_profile(mysql_read, mysql_write, uid, profile_info, token, timestamp,
                   yandexuid, subscription_id=None, allow_auth=None, refresh_token=None,
                   blackbox=None):
    profile_creator = ProfileCreator(
        mysql_read,
        mysql_write,
        uid,
        profile_info,
        token,
        timestamp,
        yandexuid,
        subscription_id,
        allow_auth,
        refresh_token,
        blackbox,
    )
    profile_creator.check_profile_possible()

    if profile_info['provider']['code'] == Kinopoisk.code:
        profile_creator.remove_blocking_kp_profiles()

    return profile_creator.create()


class ProfileCreator(object):
    def __init__(self, mysql_read, mysql_write, uid, social_userinfo, token,
                 timestamp, yandexuid=None, subscription_id=None, allow_auth=None,
                 refresh_token=None, blackbox=None):
        """
        * social_userinfo

            Информация об аккаунте в социальной сети

            Словарь с ключами provider, userid, username, birthday, firstname,
            lastname, nickname, email, phone, country, city, gender, business.

        * timestamp

            Unixtime когда таска завершилась

        * yandexuid
        """
        self._mysql_read = mysql_read
        self._mysql_write = mysql_write
        self._blackbox = blackbox

        self._uid = uid

        self._social_userinfo = social_userinfo
        self._token = token
        self._refresh_token = refresh_token

        self._timestamp = timestamp
        self._yandexuid = yandexuid
        self._allow_auth = allow_auth
        self._subscription_id = subscription_id

        self._provider = None

    def check_profile_possible(self):
        self._check_social_userinfo()

        provider = self._get_provider()

        if provider['code'] == Yandex.code:
            self._check_yandex_profile_restrictions()

        if provider['code'] == Kinopoisk.code:
            self._check_kinopoisk_profile_restrictions()

    def _check_master_slave_profile_restrictions(self, slave_checker):
        master_uid = str(self._uid)
        slave_uid = self._social_userinfo['userid']

        if master_uid == slave_uid:
            raise _MasterMatchesSlaveProfileCreationError()

        accounts, unknown_uids = get_many_accounts_by_uids(
            uids=[master_uid, slave_uid],
            blackbox_builder=self._blackbox,
            userinfo_args=dict(attributes=[], dbfields=[]),
        )
        if unknown_uids:
            raise AccountNotFoundProfileCreationError()
        master_account, slave_account = accounts
        if str(master_account.uid) == str(slave_uid):
            master_account, slave_account = slave_account, master_account

        if not (
            master_account.is_neophonish or
            master_account.is_normal or
            master_account.is_lite or
            master_account.is_social or
            master_account.is_pdd
        ):
            raise _MasterAccountTypeInvalidProfileCreationError()
        if not slave_checker(slave_account):
            raise _SlaveAccountTypeInvalidCreationError()

    def remove_blocking_kp_profiles(self):
        """
        Если создаем профиль для Кинопоиска, то удаляем все старые связки.
        Кинопоиск сам следит за уникальностью связок.
        """
        self._check_social_userinfo()

        execute(
            self._mysql_write,
            delete(pt).where(
                and_(
                    pt.c.provider_id == self._get_provider()['id'],
                    pt.c.userid == str(self._social_userinfo['userid']),
                    pt.c.uid != self._uid,
                ),
            ),
        )
        execute(
            self._mysql_write,
            delete(pt).where(
                and_(
                    pt.c.provider_id == self._get_provider()['id'],
                    pt.c.userid != str(self._social_userinfo['userid']),
                    pt.c.uid == self._uid,
                ),
            ),
        )

    def _check_kinopoisk_profile_restrictions(self):
        self._check_master_slave_profile_restrictions(slave_checker=lambda slave_account: slave_account.is_kinopoisk)

    def _check_yandex_profile_restrictions(self):
        self._check_master_slave_profile_restrictions(slave_checker=lambda slave_account: slave_account.is_phonish)

    def create(self):
        self._check_social_userinfo()
        old_profile = self.find_profile()
        new_profile = self._create_or_update_profile(old_profile)
        self._update_businessid_to_userid_mapping(old_profile, new_profile)
        if self._token:
            self._create_or_update_token(new_profile)
        self._create_or_update_subscription(new_profile)
        self._create_or_update_person(new_profile)
        return new_profile.id

    def _check_social_userinfo(self):
        if not self._social_userinfo.get('userid'):
            raise _ProviderUseridUnknownProfileCreationError()

        provider = self._get_provider()
        if not provider:
            raise _ProviderUnknownProfileCreationError()

    def find_profile(self):
        profiles_query = (
            select([pt])
            .where(get_profile_getting_query_by_profile_info(self._social_userinfo, self._uid))
        )
        profiles = execute(self._mysql_read, profiles_query).fetchall()

        if len(profiles) == 2:
            # Эта ситуация может возникнуть только так:
            # * Пользователь пришел к нам до переключения на вторую версию приложением А, мы создали профиль.
            # * Пользователь забрал у нас доступ в соц сети, профиль в базе лежит.
            # * Мы переключились на версию 2.
            # * Пользователь пришел к нам приложением Б, у него теперь профиль с business_userid.
            # * Пользователь снова приходит приложением А.
            # Теперь у пользователя два профиля, соответствующих одному соц пользователю,
            # старый нужно удалить и перенести его allow_auth и все его токены на новый профиль.
            #
            # 5 апреля 2017, осталось 1 867 883 таких профилей.
            userid_type, _ = parse_userid(profiles[0].userid)
            if userid_type == USERID_TYPE_BUSINESS:
                business_profile, simple_profile = profiles
            else:
                simple_profile, business_profile = profiles

            profile = _merge_conflicting_profiles(self._mysql_write, business_profile, simple_profile)
        elif len(profiles) == 1:
            profile = profiles[0]
        else:
            profile = None

        return profile

    def _create_or_update_profile(self, profile):
        provider = self._get_provider()
        creation_time = self._get_serialized_timestamp()

        values = {
            'provider_id': provider['id'],
            'userid': str(self._social_userinfo['userid']),
            'uid': self._uid,
            'created': creation_time,
            'confirmed': creation_time,
            'verified': creation_time,
            'username': self._social_userinfo.get('username', ''),
            'referer': 0,
            'yandexuid': self._yandexuid or '',
        }

        if profile is not None:
            values['allow_auth'] = self._allow_auth if self._allow_auth is not None else profile['allow_auth']
        else:
            values['allow_auth'] = self._allow_auth if self._allow_auth is not None else 0

        business_userid = self._get_business_userid()
        if business_userid:
            values['userid'] = business_userid

        upd_values = values.copy()
        for key in [
            'created',
            'provider_id',
            'uid',
            'userid',
        ]:
            del upd_values[key]

        if profile is None:
            insert_result = execute(
                self._mysql_write,
                insert_with_on_duplicate_key_update(pt, upd_values).values(**values),
            )

            if insert_result.lastrowid == 0:
                logger.warning('Insert.profile_id == 0')
                raise DatabaseFailedProfileCreatetionError()
            if insert_result.lastrowid is None:
                logger.warning('Insert.profile_id is None')
                raise DatabaseFailedProfileCreatetionError()

            new_profile_id = insert_result.lastrowid
        else:
            new_profile_id = profile.profile_id
            execute(
                self._mysql_write,
                update(pt).values(**upd_values).where(pt.c.profile_id == new_profile_id),
            )

        return _ProfileResultProxy(id=new_profile_id)

    def _update_businessid_to_userid_mapping(self, old_profile, new_profile):
        business_userid = self._get_business_userid()
        if business_userid:
            update_business_userid_map(
                self._mysql_read,
                self._mysql_write,
                new_profile.id,
                business_userid,
                str(self._social_userinfo['userid']),
                # Если только что создали профиль, то userid там уже новый, составной.
                old_profile.userid if old_profile else business_userid,
                self._token.application_id,
            )

    def _create_or_update_token(self, profile):
        new_token = self._token
        new_token.uid = self._uid
        new_token.profile_id = profile.id
        timestamp = datetime.fromtimestamp(self._timestamp)
        new_token.created = timestamp
        new_token.verified = timestamp
        new_token.confirmed = timestamp

        old_token = find_token_by_value_for_account(
            self._uid,
            new_token.application_id,
            new_token.value,
            db=self._mysql_read,
            application_required=False,
        )
        if old_token:
            old_token.update_with_reissued_token(new_token)
            self._token = old_token
        save_token(self._token, db=self._mysql_write)

        if self._refresh_token is not None:
            self._refresh_token.token_id = self._token.token_id
            save_refresh_token(self._refresh_token, db=self._mysql_write)

    def _create_or_update_subscription(self, new_profile):
        if self._subscription_id is not None:
            sub = execute(
                self._mysql_read,
                (
                    select([st])
                    .where(
                        and_(
                            st.c.profile_id == new_profile.id,
                            st.c.sid == self._subscription_id,
                        ),
                    )
                ),
            ).fetchone()

            if sub is None:
                execute(
                    self._mysql_write,
                    (
                        insert_with_on_duplicate_key_update(st, ['value'])
                        .values(
                            profile_id=new_profile.id,
                            sid=self._subscription_id,
                            value=1,
                        )
                    ),
                )
            elif sub['value'] == 0:
                execute(
                    self._mysql_write,
                    (
                        update(st)
                        .values(value=1)
                        .where(
                            and_(
                                st.c.profile_id == new_profile.id,
                                st.c.sid == self._subscription_id,
                            ),
                        )
                    ),
                )

    def _create_or_update_person(self, new_profile):
        person_values = {}

        for column in pert._columns:
            key = column.name
            if key in {'birthday', 'profile_id'}:
                continue
            person_values[key] = self._social_userinfo.get(key) or ''

        person_values['birthday'] = self._social_userinfo.get('birthday', '0000-00-00')

        person_values['profile_id'] = new_profile.id

        person = execute(
            self._mysql_read,
            select([pert]).where(pert.c.profile_id == new_profile.id),
        ).fetchone()

        upd_person_values = person_values.copy()
        del upd_person_values['profile_id']

        if person is None:
            execute(
                self._mysql_write,
                insert_with_on_duplicate_key_update(pert, upd_person_values).values(**person_values),
            )
        else:
            execute(
                self._mysql_write,
                update(pert).values(**upd_person_values).where(pert.c.profile_id == new_profile.id),
            )

    def _get_business_userid(self):
        return get_business_userid_from_profile_info(self._social_userinfo)

    def _get_serialized_timestamp(self):
        return serialize_unixtime(self._timestamp)

    def _get_provider(self):
        if not self._provider:
            self._provider = providers.get_provider_info_by_name(self._social_userinfo['provider']['code'])
        return self._provider


_ProfileResultProxy = namedtuple('_ProfileResultProxy', 'id')
