# -*- coding: utf-8 -*-

import logging

from passport.backend.social.common.application import (
    Application,
    ApplicationDatabaseReader,
)
from passport.backend.social.common.db.utils import get_slave_engine
from passport.backend.social.common.providers.provider import get_provider_code_to_class_mapping
from passport.backend.social.common.social_config import social_config


logger = logging.getLogger(__name__)


class Providers(object):
    _instance = None
    _initialized = False
    __providers = {}

    def __new__(cls, *more):
        if cls._instance is None:
            cls._instance = super(Providers, cls).__new__(cls, *more)
        return cls._instance

    def init(self):
        self.init_from_dict(
            dict(
                providers=social_config.providers,
            )
        )

    def init_from_dict(self, conf):
        self.__providers = {}

        for provider in conf['providers']:
            self.__providers[provider['code']] = provider
            self.__providers[provider['name']] = provider
            self.__providers[provider['id']] = provider
            self.__providers[str(provider['id'])] = provider

        self._initialized = True

    def get_application_for_provider(self, provider_code, tld=None):
        provider = self.__providers.get(provider_code)

        if provider is None:
            return

        apps = ApplicationDatabaseReader(get_slave_engine()).load_by_provider_ids([provider['id']])
        apps = [Application().parse(a) for a in apps]

        app_for_provider = None
        if tld:
            for app in apps:
                if app.tld and tld in app.tld:
                    app_for_provider = app
                    break

        if not app_for_provider:
            for app in apps:
                if app.default:
                    app_for_provider = app
                    break

        return app_for_provider

    def get_provider_info_by_name(self, provider_name):
        return self.__providers.get(provider_name)

    def get_provider_info_by_id(self, provider_id, include_provider_class=False):
        rv = self.__providers.get(provider_id)
        if rv is not None and include_provider_class:
            rv = dict(rv)
            mapping = get_provider_code_to_class_mapping()
            rv['class'] = mapping[rv['code']]
        return rv

    def get_application_by_id(self, app_id):
        apps, _ = self.get_many_applications_by_ids([app_id])
        if apps:
            return apps[0]

    def get_many_applications_by_ids(self, app_id_list):
        app_dicts = (
            ApplicationDatabaseReader(get_slave_engine())
            .load_by_application_ids(app_id_list)
        )

        apps = list()
        unknown_ids = set(app_id_list)
        for app_dict in app_dicts:
            app = Application().parse(app_dict)
            apps.append(app)
            # Т.к. числовой идентификатор приложения могли передать строкой,
            # здесь нужно удалить оба варианта.
            unknown_ids.discard(app.identifier)
            unknown_ids.discard(str(app.identifier))
        unknown_ids = list(unknown_ids)
        return apps, unknown_ids

    def get_application_by_name(self, app_name):
        apps, _ = self.get_many_applications_by_names([app_name])
        if apps:
            return apps[0]

    def get_many_applications_by_names(self, app_name_list):
        app_dicts = (
            ApplicationDatabaseReader(get_slave_engine())
            .load_by_application_names(app_name_list)
        )

        apps = list()
        unknown_names = set(app_name_list)
        for app_dict in app_dicts:
            app = Application().parse(app_dict)
            apps.append(app)
            unknown_names.remove(app.name)
        unknown_names = list(unknown_names)
        return apps, unknown_names

    def get_application_by_provider_app_id(self, provider_id, provider_app_id):
        app_dicts = (
            ApplicationDatabaseReader(get_slave_engine())
            .load_by_provider_client_ids([(provider_id, provider_app_id)])
        )
        if app_dicts:
            return Application().parse(app_dicts[0])

    def get_many_applications_by_provider(self, provider_code):
        provider = self.__providers.get(provider_code)

        if provider is None:
            return list()

        apps = ApplicationDatabaseReader(get_slave_engine()).load_by_provider_ids([provider['id']])
        return [Application().parse(a) for a in apps]

    @property
    def providers(self):
        return self.__providers


providers = Providers()


def get_profile_addresses(provider_id, userid=None, username=None, uid=None, profile_id=None):
    provider_info = providers.get_provider_info_by_id(
        provider_id, include_provider_class=True
    )
    if not provider_info:
        logger.debug('Unable to generate profile links for unknown provider: %s' % provider_id)
        return list()
    return provider_info['class'].profile_link(
        userid=userid,
        username=username,
        profile_id=profile_id,
        uid=uid,
    )
