# -*- coding: utf-8 -*-

from abc import ABCMeta
from collections import namedtuple

from passport.backend.core.lazy_loader import (
    lazy_loadable,
    LazyLoader,
)
from passport.backend.social.common.exception import (
    ApplicationUnknown,
    ProviderUnknown,
)
from passport.backend.social.common.misc import (
    discover_modules,
    discover_subclasses,
    parse_userid,
    USERID_TYPE_SIMPLE,
)
import passport.backend.social.common.providers


ProviderRequest = namedtuple('ProviderRequest', 'url headers body')


def get_provider_code_to_class_mapping():
    return LazyLoader.get_instance('provider_code_to_class_mapping')


@lazy_loadable('provider_code_to_class_mapping')
def _discover_providers():
    providers = dict()
    for module in discover_modules(passport.backend.social.common.providers):
        for provider_cls in discover_subclasses(module, Provider):
            if provider_cls.code:
                providers[provider_cls.code] = provider_cls
    return providers


class Provider(object):
    # code - двухбуквенный код провайдера
    # exchange - параметр обычно используемый для обмена на access token
    __slots__ = ('id', 'name', 'code', 'exchange', 'scope', 'display', 'secret', 'profile', '_app')
    __metaclass__ = ABCMeta  # для назначения абстрактных методов

    profile_links = {}

    def __init__(self, **kwargs):
        # инициализируем все слоты None, если изначально не установлены аттрибуты класса
        for attr in self.__slots__:
            if not hasattr(self, attr):
                setattr(self, attr, None)
        for k, v in kwargs.items():
            setattr(self, k, v)

    @property
    def app(self):
        return self._app

    @app.setter
    def app(self, app):
        self._app = app

    def __unicode__(self):
        return self.name

    def __str__(self):
        return self.__unicode__()

    @staticmethod
    def create(code=None, app=None, **kwargs):
        """
        Абстрактная фабрика для создания конкретных объектов провайдеров
        """
        mapping = get_provider_code_to_class_mapping()
        if code and code in mapping:
            if not app:
                raise ApplicationUnknown('No data about application for provider %s' % code)
            kwargs['app'] = app
            provider_obj = mapping[code](**kwargs)
            return provider_obj
        raise ProviderUnknown('Cannot create provider with code=%s' % code)

    @classmethod
    def profile_link(cls, userid=None, username=None, profile_id=None, uid=None):
        res = []
        if not hasattr(cls, 'profile_links'):
            return res

        if userid and 'id' in cls.profile_links:
            userid_type, _ = parse_userid(userid)
            if userid_type == USERID_TYPE_SIMPLE:
                res.append(cls.profile_links['id'] % userid)
        if username and 'username' in cls.profile_links:
            res.append(cls.profile_links['username'] % username)
        return res
