# -*- coding: utf-8 -*-

from __future__ import unicode_literals

import logging
from time import sleep

from passport.backend.core.lazy_loader import LazyLoader
from passport.backend.core.logging_utils.loggers.graphite import GraphiteLogger
from passport.backend.social.common.chrono import now
from passport.backend.social.common.context import request_ctx
from passport.backend.social.common.misc import trim_message
from passport.backend.social.common.social_config import social_config
import redis


logger = logging.getLogger('redis')
graphite = GraphiteLogger(logging.getLogger('graphite.redis'))

REDIS_DB_INDEX = 0

REDIS_OP_TYPE_READ = 'read'
REDIS_OP_TYPE_WRITE = 'write'

SUCCESS_REDIS_RESPONSE = 'success'
TIMEOUT_REDIS_RESPONSE = 'timeout'
FAIL_REDIS_RESPONSE = 'fail'


class RedisError(Exception):
    pass


# TODO - отчасти это копипаста из yandex-passport-core, который пока нельзя
# использовать как минимум по причине Python2.6
# Когда-нибудь стоит вынести это в отдельный модуль и пакет.
class RedisClient(object):
    """
    Клиент для работы с redis. Создает пул соединений один для всех инстансов клиента.
    Пул инициализируется вызовом RedisClient.init(config)
    """
    _pool_read = None
    _pool_write = None
    _master = None
    _slave = None
    _initialized = False

    @property
    def initialized(self):
        return self._initialized

    def __init__(self):
        if not self._initialized:
            raise RedisError('Uninitialized RedisClient can not be instantiated.')

    @classmethod
    def init(cls):
        """
        Создает пулы для чтения и записи..
        Returns: None
        """

        cls._pool_read = redis.ConnectionPool(
            host=social_config.redis_slave_host,
            port=social_config.redis_slave_port,
            db=REDIS_DB_INDEX,
            password=social_config.redis_slave_password,
            max_connections=social_config.redis_pool_size,
            socket_timeout=social_config.redis_connection_timeout,
        )
        cls._pool_write = redis.ConnectionPool(
            host=social_config.redis_master_host,
            port=social_config.redis_master_port,
            db=REDIS_DB_INDEX,
            password=social_config.redis_master_password,
            max_connections=social_config.redis_pool_size,
            socket_timeout=social_config.redis_connection_timeout,
        )
        cls._master = redis.Redis(connection_pool=cls._pool_read)
        cls._slave = redis.Redis(connection_pool=cls._pool_write)
        cls._initialized = True

    @classmethod
    def stop(cls):
        """
        Закрываем соединения
        Returns: None
        """

        if cls._pool_read:
            cls._pool_read.disconnect()
        if cls._pool_write:
            cls._pool_write.disconnect()

        cls._initialized = False

    def _execute(self, _redis, op_type, op):
        if not self._initialized:
            raise Exception('Attempt to use an uninitialized RedisClient.')

        retries = social_config.redis_reconnect_retries
        connection_info = _redis.connection_pool.connection_kwargs

        while retries > 0:
            start_time = now.f()
            try:
                result = op(_redis)
            except redis.RedisError as e:
                duration = now.f() - start_time
                retries -= 1

                termination = retries <= 0
                self._log_error(e, retries, connection_info, op_type, duration, termination)

                if retries > 0:
                    sleep(social_config.redis_reconnect_timeout)
            else:
                duration = now.f() - start_time
                retries -= 1
                self._log_success(retries, connection_info, op_type, duration, termination=True)
                return result

        raise RedisError(e)

    def get(self, key):
        logger.debug('get called for key: "%s"', key)
        return self._execute(
            self._slave,
            REDIS_OP_TYPE_READ,
            lambda conn: conn.get(key),
        )

    def expire(self, key, time):
        logger.debug('expire called for key: "%s", seconds_count=%s', key, time)
        return self._execute(
            self._master,
            REDIS_OP_TYPE_WRITE,
            lambda conn: conn.expire(key, time),
        )

    def hmset(self, key, data):
        logger.debug('hmset called for key: "%s"', key)
        return self._execute(
            self._master,
            REDIS_OP_TYPE_WRITE,
            lambda conn: conn.hmset(key, data),
        )

    def hgetall(self, key):
        logger.debug('hgetall called for key: "%s"', key)
        return self._execute(
            self._slave,
            REDIS_OP_TYPE_READ,
            lambda conn: conn.hgetall(key),
        )

    def delete(self, *keys):
        logger.debug('delete called for keys "%s"' % keys)
        return self._execute(
            self._master,
            REDIS_OP_TYPE_WRITE,
            lambda conn: conn.delete(*keys),
        )

    def set(self, key, value):
        logger.debug('set called for key "%s", value "%s"' % (key, value))
        return self._execute(
            self._master,
            REDIS_OP_TYPE_WRITE,
            lambda conn: conn.set(key, value),
        )

    def rpush(self, key, *values):
        return self._execute(
            self._master,
            REDIS_OP_TYPE_WRITE,
            lambda conn: conn.rpush(key, *values),
        )

    def ltrim(self, key, start, end):
        return self._execute(
            self._master,
            REDIS_OP_TYPE_WRITE,
            lambda conn: conn.ltrim(key, start, end),
        )

    def lrange(self, key, start, end):
        return self._execute(
            self._slave,
            REDIS_OP_TYPE_READ,
            lambda conn: conn.lrange(key, start, end),
        )

    # остальные методы добавляются по мере необходимости

    def _log_error(self, exception, retries_left, connection_info, query_type,
                   duration, termination):
        retry_no = social_config.redis_reconnect_retries - retries_left
        logger.warn(
            'Redis error: host %s:%s "%s"; Attempt number: %d of %d' % (
                connection_info['host'],
                connection_info['port'],
                trim_message(exception),
                retry_no,
                social_config.redis_reconnect_retries,
            ),
        )

        log_record = self._build_common_graphite_record(
            retries_left,
            connection_info,
            query_type,
            duration,
            termination,
        )
        if isinstance(exception, redis.TimeoutError):
            log_record['response'] = TIMEOUT_REDIS_RESPONSE
        else:
            log_record['response'] = FAIL_REDIS_RESPONSE
        graphite.log(**log_record)

    def _log_success(self, retries_left, connection_info, query_type, duration, termination):
        log_record = self._build_common_graphite_record(
            retries_left,
            connection_info,
            query_type,
            duration,
            termination,
        )
        log_record.update(response=SUCCESS_REDIS_RESPONSE)
        graphite.log(**log_record)

    def _build_common_graphite_record(self, retries_left, connection_info, query_type,
                                      duration, termination):
        if termination:
            retries_left = 0
        return dict(
            tskv_format='social-redis-log',
            srv_hostname=connection_info['host'],
            service='redis',
            duration=duration,
            retries_left=retries_left,
            request_id=request_ctx.request_id,
            query_type=query_type,
        )


def get_redis():
    return LazyLoader.get_instance('redis')
