# -*- coding: utf-8 -*-

import logging


class Settings(object):
    def __init__(self):
        self._settings = None
        self._options = dict()

    def __getattr__(self, name):
        if name in self._options:
            return self._options[name]

        if self._settings is not None:
            return getattr(self._settings, name)
        else:
            return _LazyObject(self, name)

    def __dir__(self):
        return self._options.keys() + dir(self._settings)

    def configure(self, settings=None, **options):
        if self.configured:
            logging.debug('Settings already configured.')

        self._settings = settings
        self._options.update(**options)

    @property
    def configured(self):
        return bool(self._settings or self._options)

    def property(self, name):
        return _SettingsProperty(self, name)


class _LazyObject(object):
    def __init__(self, host_object, name):
        super(_LazyObject, self).__setattr__('_host_object', host_object)
        super(_LazyObject, self).__setattr__('_name', name)

    def __repr__(self):
        class_name = type(self).__name__
        host_object_name = type(self._host_object).__name__
        return "<%s for '%s' in %s>" % (class_name, self._name, host_object_name)

    def __getattr__(self, name):
        real_object = getattr(self._host_object, self._name)
        assert not isinstance(real_object, _LazyObject)
        return getattr(real_object, name)

    def __setattr__(self, name, value):
        real_object = getattr(self._host_object, self._name)
        assert not isinstance(real_object, _LazyObject)
        return setattr(real_object, name, value)

    def __str__(self):
        real_object = getattr(self._host_object, self._name)
        assert not isinstance(real_object, _LazyObject)
        return real_object.__str__()

    def __getitem__(self, key):
        real_object = getattr(self._host_object, self._name)
        assert not isinstance(real_object, _LazyObject)
        return real_object.__getitem__(key)

    def __setitem__(self, key, value):
        real_object = getattr(self._host_object, self._name)
        assert not isinstance(real_object, _LazyObject)
        return real_object.__setitem__(key, value)


class _SettingsProperty(object):
    def __init__(self, settings, name):
        self._settings = settings
        self._name = name

    def __get__(self, obj, obj_type=None):
        return getattr(self._settings, self._name)
