# -*- coding: utf-8 -*-

from __future__ import unicode_literals

import codecs
from datetime import datetime
from errno import EBADF
from logging import (
    Filter as LoggingFilter,
    Formatter as LoggingFormatter,
    getLogger as logging_get_logger,
)
from os import close as os_close
from traceback import extract_tb

from passport.backend.core.logging_utils.loggers.tskv import (
    TskvLogEntry,
    TskvLogger,
)
from passport.backend.social.common.chrono import now
from passport.backend.social.common.misc import (
    dump_to_json_string,
    request_context_to_application_info,
)
from passport.backend.utils.string import encoding_aware_split_bytes
from passport.backend.utils.time import (
    get_unixtime,
    unixtime_to_datetime,
)
from ylog.handlers import OSFileHandler as YlogOsFileHandler


def close_logging_handlers(logging_conf):
    known_loggers = set([logging_get_logger()])
    for logger_name in logging_conf.get('loggers', dict()):
        known_loggers.add(logging_get_logger(logger_name))

    all_handlers = set()
    for logger in known_loggers:
        all_handlers |= set(logger.handlers)

    known_handler_names = set(logging_conf.get('handlers', dict()).keys())
    known_handlers = set([h for h in all_handlers if h.name in known_handler_names])

    for logger in known_loggers:
        for handler in logger.handlers[:]:
            if handler in known_handlers:
                logger.removeHandler(handler)

    for handler in known_handlers:
        handler.close()


class SocialFormatter(LoggingFormatter):
    MAX_LINE_BYTES = 4096
    MIN_MESSAGE_BYTES = 1000

    def __init__(
        self,
        context,
        fmt='[%(asctime)s][%(levelname)s][%(request_id)s][%(task_id)s][%(provider_code)s][%(application_name)s] %(message)s',
        datefmt=None,
        logtype=None,
        encoding='utf-8',
    ):
        self._encoding = encoding
        self._context = context
        self._logtype = logtype
        self._decoder = codecs.getincrementaldecoder(self._encoding)()
        super(SocialFormatter, self).__init__(fmt, datefmt)

    def format(self, record):
        self._add_request_context_to_record(record)
        record.logtype = self._logtype or '-'

        if isinstance(record.msg, bytes):
            record.msg = record.msg.decode(self._encoding)
        _str = super(SocialFormatter, self).format(record)

        _str = self._escaped(_str)
        if len(_str.encode(self._encoding)) <= self.MAX_LINE_BYTES:
            return _str

        return '\n'.join(self._split_long_string_to_atomic_substrings(_str, self._escaped(record.message)))

    def _split_long_string_to_atomic_substrings(self, _str, message):
        """
        Разбивает очень длинную запись в лог на несколько строк, сохраняя общий
        префикс.

        Вызов write в Linux гарантирует атомарную запись в файл, только если
        длина данных не превышает 4КБ

        Параметры
            _str — вся строка целиком
            message — только часть строки с сообщением
        """
        prefix = _str[:-len(message)]
        self._decoder.reset()

        lines = encoding_aware_split_bytes(
            message.encode(self._encoding),
            # Припасём 1 байт для перевода на новую строку
            byte_length=max(self.MIN_MESSAGE_BYTES, self.MAX_LINE_BYTES - len(prefix.encode(self._encoding))) - 1,
            incremental_decoder=self._decoder,
            align_byte=b' ',
        )
        return [prefix + line for line in lines]

    def _escaped(self, s):
        return s.replace('\n', r'\n').replace('\r', r'\r')

    def _add_request_context_to_record(self, record):
        for attr in ('request_id', 'task_id', 'handler_id'):
            value = getattr(self._context, attr, None) or '-'
            setattr(record, attr, value)

        app_info = request_context_to_application_info(self._context)
        record.provider_code = app_info['provider_code'] or '-'
        record.application_name = app_info['application_name'] or '-'


class OsFileHandler(YlogOsFileHandler):
    def close(self):
        try:
            os_close(self.fd)
        except OSError as e:
            if e.errno != EBADF:
                raise
        super(OsFileHandler, self).close()

    def createLock(self):
        self.lock = None


class ExceptionFilter(LoggingFilter):
    """
    Фильтр пропускает только сообщения с информацией об исключениях.
    """
    def filter(self, record):
        is_allowed = super(ExceptionFilter, self).filter(record)
        if is_allowed:
            is_allowed = 1 if record.exc_info else 0
        return is_allowed


class LevelFilter(LoggingFilter):
    """
    Фильтр пропускает только сообщения с заданным уровнем важности.
    """
    def __init__(self, levels, name=''):
        super(LevelFilter, self).__init__(name)
        self._levels = set(levels)

    def filter(self, record):
        is_allowed = super(LevelFilter, self).filter(record)
        if is_allowed:
            if record.levelname in self._levels:
                is_allowed = 1
            else:
                is_allowed = 0
        return is_allowed


class ExceptionFormatter(SocialFormatter):
    """
    Форматтер подставляет в поле exception информацию о исключении:
        тип-исключения путь-в-фс-до-модуля:номер-raise-строки str(исключение)
    """
    def __init__(self,
                 context,
                 fmt='%(asctime)s unixtime=%(created)s %(logtype)s %(request_id)s %(exception)s',
                 **kwargs):
        super(ExceptionFormatter, self).__init__(context, fmt=fmt, **kwargs)

    def format(self, record):
        self._add_request_context_to_record(record)

        record.message = record.getMessage()
        record.logtype = self._logtype or '-'
        if self.usesTime():
            record.asctime = self.formatTime(record, self.datefmt)
        record.exception = self.formatException(record.exc_info)

        _str = self._fmt % record.__dict__

        return self._escaped(_str)

    def formatException(self, exc_info):
        if not (exc_info and exc_info[0]):
            return '-'
        e_type, exc, tb = exc_info
        frame_info_stack = extract_tb(tb)
        frame_info = frame_info_stack[-1]
        module_path = frame_info[0]
        line_no = frame_info[1]
        return '%s %s:%s %s' % (e_type.__name__, module_path, line_no, exc)


class WarningFormatter(SocialFormatter):
    def __init__(self,
                 context,
                 fmt='%(asctime)s unixtime=%(created)s %(logtype)s %(request_id)s %(handler_id)s %(provider_code)s %(application_name)s * %(message)s',
                 **kwargs):
        super(WarningFormatter, self).__init__(context, fmt=fmt, **kwargs)


class SocialTskvLogEntry(TskvLogEntry):
    DATETIME_FORMAT = '%d/%b/%Y:%H:%M:%S'

    def __init__(self, **params):
        params.setdefault('unixtime', get_unixtime())
        params['timestamp'] = self.format_timestamp(params['unixtime'])
        super(SocialTskvLogEntry, self).__init__(**params)

    @staticmethod
    def truncate_str(text):
        return text

    @classmethod
    def format_timestamp(cls, timestamp=None):
        if timestamp is None:
            timestamp = now()
        elif not isinstance(timestamp, datetime):
            timestamp = unixtime_to_datetime(timestamp)
        return timestamp.strftime(cls.DATETIME_FORMAT)


class SocialTskvLogger(TskvLogger):
    entry_class = SocialTskvLogEntry

    def log_event(self, event):
        self.log(**event.asdict())


class BindingLogEntry(SocialTskvLogEntry):
    def __init__(self, **params):
        params.setdefault('tskv_format', 'social-binding-log')
        super(BindingLogEntry, self).__init__(**params)


class BindingLogger(SocialTskvLogger):
    default_logger_name = 'bindings_log'
    entry_class = BindingLogEntry


class BindingCreatedStatboxEvent(object):
    def __init__(self, master_uid, slave_userid, provider_code):
        self.master_uid = master_uid
        self.slave_userid = slave_userid
        self.provider_code = provider_code

    def asdict(self):
        return dict(
            action='binding_created',
            master_uid=self.master_uid,
            slave_userid=self.slave_userid,
            provider_code=self.provider_code,
        )


class BindingsDeletedStatboxEvent(object):
    def __init__(self, master_uid, slave_provider_userids):
        # slave_provider_userids -- список пар (provide_code, userid)

        self.master_uid = master_uid

        self.slave_provider_userids = list()
        for provider_code, provider_userid in slave_provider_userids:
            bit = (str(provider_code), str(provider_userid))
            self.slave_provider_userids.append(bit)

    def asdict(self):
        return dict(
            action='bindings_deleted',
            master_uid=self.master_uid,
            slave_provider_userids=dump_to_json_string(self.slave_provider_userids, minimal=True),
        )
