# -*- coding: utf-8 -*-

from functools import partial
import logging
from sys import exc_info
from traceback import (
    extract_stack,
    extract_tb,
    format_exception_only,
    format_list,
)

from gevent import get_hub
from passport.backend.social.common.context import request_ctx
from passport.backend.social.common.misc import trim_message


logger = logging.getLogger(__name__)


def threadpool_apply(f, args=None, kwargs=None, traceback_logger=None):
    args = args or tuple()
    kwargs = kwargs or dict()

    f = partial(_exc_to_retval, f)
    # gevent создаёт сопроцесс с чистым стешом, чтобы реализовать
    # наследование стеша нужно его заново наполнить данными отцовского
    # процесса.
    f = partial(_call_with_stash, f, request_ctx.save())
    retval, exception_info = _start_and_wait(f, args, kwargs)
    if exception_info and exception_info[0]:
        exc_type, exc_value, exc_traceback = exception_info
        if traceback_logger:
            stack_trace = _compose_stack_trace(exc_traceback)
            formatted_traceback = _format_thread_traceback(exc_type, exc_value, stack_trace)
            traceback_logger.debug(trim_message(formatted_traceback, cut=False))
        if isinstance(exc_value, BaseException):
            raise exc_value
        else:
            raise exc_type(exc_value)
    return retval


def _call_with_stash(func, stash_dict, *args, **kwargs):
    request_ctx.load(stash_dict)
    try:
        return func(*args, **kwargs)
    finally:
        # Чистим память гринлета, чтобы не было утечек
        request_ctx.clear()


def _exc_to_retval(f, *args, **kwargs):
    try:
        retval = f(*args, **kwargs)
        return [retval, None]
    except:
        return [None, exc_info()]


def _start_and_wait(f, args, kwargs):
    return get_hub().threadpool.apply(f, args=args, kwds=kwargs)


def _compose_stack_trace(exc_traceback):
    t1 = extract_tb(exc_traceback)
    t2 = extract_stack()
    return t2[:-2] + t1[1:]


def _format_thread_traceback(exc_type, exc_value, stack_trace):
    lines = [
        'Greenlet failed: %s\n' % exc_type.__name__,
        'Traceback (most recent call last):\n',
    ]
    lines += format_list(stack_trace)
    lines += format_exception_only(exc_type, exc_value)
    return ''.join(lines)
