# -*- coding: utf-8 -*-

from __future__ import unicode_literals

import logging

from passport.backend.social.common.builders.kolmogor import (
    get_kolmogor,
    KolmogorPermanentError,
    KolmogorTemporaryError,
)
from passport.backend.social.common.counter import (
    get_custom_consumer_for_counters,
    ProcessRequestByConsumerCounterDescriptor,
)
from passport.backend.social.common.exception import (
    GrantsContextError,
    RateLimitExceededError,
)
from passport.backend.social.common.grants import find_all_consumers_by_context


logger = logging.getLogger(__name__)


class Throttler(object):
    def __init__(self, request_name, consumers):
        self.request_name = request_name
        self.consumers = consumers
        self._kolmogor = get_kolmogor()

    @classmethod
    def from_grants_context(cls, request_name, grants_config, grants_context):
        try:
            consumers = find_all_consumers_by_context(grants_config, grants_context)
        except GrantsContextError:
            logger.warning('Failed to find consumers in grants context')
            consumers = list()
        return cls(request_name, consumers)

    def throttle(self):
        if not self.consumers:
            raise RateLimitExceededError('Consumer is unknown')

        counter_desc_list = list()
        for consumer in self.consumers:
            consumer = get_custom_consumer_for_counters(consumer)
            counter_desc = ProcessRequestByConsumerCounterDescriptor(self.request_name, consumer)
            if not counter_desc.limit:
                return
            counter_desc_list.append(counter_desc)

        try:
            keys = [c.name for c in counter_desc_list]
            counters = self._kolmogor.get('socialism_short_life', keys)
        except (KolmogorPermanentError, KolmogorTemporaryError):
            logger.warning('Failed to get counters: %s' % ','.join(keys))
            return

        for counter_desc in counter_desc_list:
            counter_value = counters[counter_desc.name]
            if counter_value < counter_desc.limit:
                try:
                    self._kolmogor.inc('socialism_short_life', [counter_desc.name])
                except (KolmogorPermanentError, KolmogorTemporaryError):
                    logger.warning('Failed to update counter: %s' % counter_desc.name)
                return

        raise RateLimitExceededError(
            'Rate limit for %s on %s is exceeded' % (
                self._formatted_consumers,
                self.request_name,
            ),
        )

    @property
    def _formatted_consumers(self):
        formatted_consumers = list()
        for consumer in self.consumers:
            formatted_consumers.append(consumer)

            custom_consumer = get_custom_consumer_for_counters(consumer)
            if custom_consumer != consumer:
                formatted_consumers[-1] += ' (%s)' % custom_consumer

        return ','.join(formatted_consumers)
