# -*- coding: utf-8 -*-

from __future__ import unicode_literals

from datetime import (
    datetime,
    timedelta,
)

from passport.backend.social.common.chrono import (
    datetime_to_unixtime,
    now,
)
from passport.backend.social.common.misc import split_scope_string
from passport.backend.social.common.provider_settings import providers


class Token(object):
    _REQUIRED_CONSTRUCTOR_ARGS = {
        'token_id',
        'uid',
        'profile_id',
        # Внутренний идентификатор приложения
        'application_id',
        # Объект Application
        'application',
        'value',
        'secret',
        'scopes',
        'expired',
        'created',
        'verified',
        'confirmed',
    }

    def __init__(self, **kwargs):
        for arg_name in kwargs:
            if arg_name not in self._REQUIRED_CONSTRUCTOR_ARGS:
                raise TypeError("__init__() got an unexpected keyword argument '%s'" % arg_name)

        self._scopes = set()
        self._secret = None
        self._expired = None

        kwargs.setdefault('token_id', None)
        kwargs.setdefault('profile_id', None)
        kwargs.setdefault('uid', None)
        for arg_name in self._REQUIRED_CONSTRUCTOR_ARGS:
            setattr(self, arg_name, kwargs.get(arg_name))

    def set_scopes(self, value):
        if value is None:
            value = set()
        elif isinstance(value, basestring):
            value = set(split_scope_string(value))
        else:
            value = set(value)
        self._scopes = value

    def get_scopes(self):
        return self._scopes

    scopes = property(get_scopes, set_scopes)

    def set_secret(self, value):
        if value == '':
            value = None
        self._secret = value

    def get_secret(self):
        return self._secret

    secret = property(get_secret, set_secret)

    def set_expired(self, value):
        if value is not None and not isinstance(value, datetime):
            value = datetime.fromtimestamp(float(value))
        self._expired = value

    def get_expired(self):
        return self._expired

    expired = property(get_expired, set_expired)

    def update_with_reissued_token(self, other):
        """
        Метод для подновления токена, когда система авторизации выдала токен со
        старым значением.
        """
        assert self.application_id == other.application_id
        assert self.value == other.value

        # Хак нужный для работы get_token_newest, правильно было бы завести
        # новое свойство токена updated. Но пока не жмёт можно и так
        self.created = max([t for t in [self.created, other.created] if t])

        self.expired = other.expired
        self.scopes |= other.scopes
        self.profile_id = other.profile_id

    def has_every_scope(self, scopes):
        return all(s in self.scopes for s in scopes)

    def is_going_to_expire(self):
        return self.expired is not None and self.expired <= now() + timedelta(minutes=1)

    def to_dict(self):
        return {
            'token_id': self.token_id,
            'uid': self.uid,
            'profile_id': self.profile_id,
            'application_id': self.application_id,
            'value': self.value,
            'secret': self.secret,
            'scopes': self.scopes,
            'expired': self.expired,
            'created': self.created,
            'verified': self.verified,
            'confirmed': self.confirmed,
        }

    def to_json_dict(self):
        _dict = self.to_dict()

        application = self.application
        if not application:
            application = providers.get_application_by_id(self.application_id)

        if application and application.name:
            _dict['application'] = application.name
        else:
            _dict['application'] = ''

        _dict['scope'] = ' '.join(sorted(self.scopes))

        for time_name in ['expired', 'created', 'confirmed', 'verified']:
            time_value = _dict[time_name]
            formatted_time = time_value.strftime('%Y-%m-%d %H:%M:%S') if time_value is not None else None
            unixtime = datetime_to_unixtime(time_value) if time_value is not None else None
            _dict.update({time_name: formatted_time, time_name + '_ts': unixtime})

        del _dict['scopes']
        del _dict['application_id']

        return _dict

    def to_dict_for_proxy(self):
        return {
            'token_id': self.token_id,
            'application': self.application_id,
            'value': self.value,
            'secret': self.secret,
            'expires': datetime_to_unixtime(self.expired) if self.expired is not None else None,
            'scope': ','.join(sorted(self.scopes)),
        }

    def update_from_dict_for_proxy(self, dict_for_proxy):
        app_id = dict_for_proxy.get('application')
        if app_id:
            self.application_id = app_id

        value = dict_for_proxy.get('value')
        if value:
            self.value = value

        secret = dict_for_proxy.get('secret')
        if secret:
            self.secret = secret

        expires = dict_for_proxy.get('expires')
        if expires is not None:
            self.expired = datetime.fromtimestamp(expires)

        scope = dict_for_proxy.get('scope')
        if scope:
            self.scopes = split_scope_string(scope)

    @classmethod
    def from_dict_for_proxy(cls, dict_for_proxy):
        token = cls(
            application_id=None,
            value=None,
            secret=None,
            scopes=None,
            expired=None,

            created=None,
            verified=None,
            confirmed=None,
        )
        token.update_from_dict_for_proxy(dict_for_proxy)
        return token
