# -*- coding: utf-8 -*-

from __future__ import unicode_literals

import logging

from passport.backend.social.common.exception import ApplicationUnknown
from passport.backend.social.common.provider_settings import providers
from passport.backend.social.common.token.domain import Token

from .db import TokenRecord


logger = logging.getLogger(__name__)


def find_token_by_value_for_account(
    uid,
    application_id,
    value,
    db,
    application_required=True,
):
    records = TokenRecord.find_by_value_for_account(uid, application_id, value, db)
    assert 0 <= len(records) <= 1
    if not records:
        return
    token = records[0].to_model()
    _load_applications([token])
    if not application_required:
        return token
    if token.application:
        return token


def find_token_by_token_id(token_id, db):
    records = TokenRecord.find_by_token_id(token_id, db)
    assert 0 <= len(records) <= 1
    if not records:
        return
    token = records[0].to_model()
    _load_applications([token])
    if token.application:
        return token


def find_all_tokens_for_profile(profile_id, db, application_ids=None):
    records = TokenRecord.find_all_for_profile(profile_id, application_ids, db)
    tokens = [r.to_model() for r in records]
    return _load_applications(tokens)[0]


def find_all_tokens_for_profiles(profile_ids, db):
    records = TokenRecord.find_all_for_profile_ids(profile_ids, db)
    tokens = [r.to_model() for r in records]
    return _load_applications(tokens)[0]


def find_all_tokens_for_account(uid, db, application_ids=None):
    records = TokenRecord.find_all_for_account(uid, application_ids, db)
    tokens = [r.to_model() for r in records]
    return _load_applications(tokens)[0]


def save_token(token, db):
    record = TokenRecord.from_model(token)
    record.save(db)
    return record


def delete_token_by_token_id(token_id, db):
    assert token_id is not None
    return TokenRecord.delete_by_token_ids([token_id], db)


def delete_all_tokens_for_account(uid, application_ids, read_conn, write_conn):
    return TokenRecord.delete_all_for_account(uid, application_ids, read_conn, write_conn)


def build_token_from_token_data(token_in_task, uid, profile_id, timestamp):
    application = providers.get_application_by_name(token_in_task['application'])
    if not application:
        raise ApplicationUnknown()

    return Token(
        uid=uid,
        profile_id=profile_id,
        application_id=application.identifier,
        value=token_in_task['value'],
        secret=token_in_task.get('secret'),
        scopes=token_in_task.get('scope'),
        expired=token_in_task.get('expires'),
        created=timestamp,
        verified=timestamp,
        confirmed=timestamp,
    )


def build_token_dict_for_proxy(server_token, refresh_token):
    token_dict = server_token.to_dict_for_proxy()
    if refresh_token:
        refresh_token.update_token_dict_for_proxy(token_dict)
    return token_dict


def filter_tokens_granted_to_read(tokens, filter_allowed_grants_func):
    return _filter_tokens_granted_to_do_something(get_grants_to_read_token, tokens, filter_allowed_grants_func)


def filter_tokens_granted_to_use(tokens, filter_allowed_grants_func):
    return _filter_tokens_granted_to_do_something(get_grants_to_use_token, tokens, filter_allowed_grants_func)


def _filter_tokens_granted_to_do_something(get_grants_to_do_something_with_token, tokens, filter_allowed_grants_func):
    """
    Возвращает только токены, на которые у потребителя есть грант

    Параметры

    filter_allowed_grants_func

        Функция пересечёт множество всех грантов, которыми можно использовать
        данные токены, с множеством грантов, которые есть у данного потребителя.
        А затем вернёт полученное пересечение.

        filter_allowed_grants_func(grants: set[str]) -> set[str]
    """
    grants = set()
    token_grants = dict()
    for token in tokens:
        token_grants[token] = get_grants_to_do_something_with_token(token)
        grants.update(token_grants[token])
    grants = filter_allowed_grants_func(grants)

    granted_tokens = list()
    skipped_apps = set()
    for token in tokens:
        if token_grants[token] & grants:
            granted_tokens.append(token)
        else:
            skipped_apps.add(token.application.name)

    if skipped_apps:
        formatted_skipped_apps = ', '.join(map(str, skipped_apps))
        logger.debug('Skipped tokens beacause of missing grants for applications: %s' % formatted_skipped_apps)

    return granted_tokens


def get_grants_to_read_token(token=None, application=None):
    return _get_grants_to_something_token('read', token=token, application=application)


def get_grants_to_update_token(token=None, application=None):
    return _get_grants_to_something_token('update', token=token, application=application)


def get_grants_to_use_token(token=None, application=None):
    return (
        _get_grants_to_something_token('use', token=token, application=application) |
        _get_grants_to_something_token('read', token=token, application=application)
    )


def _get_grants_to_something_token(something, token=None, application=None):
    grants = {'no-cred-%s-token' % something}

    if token is not None:
        application = token.application

    grants.update({'no-cred-%s-token-application:%s' % (something, application.name)})

    if application.group_id:
        grants.update({'no-cred-%s-token-application-group:%s' % (something, application.group_id)})

    from passport.backend.social.common.builders.blackbox import authenticated

    if authenticated():
        grants.update({'has-cred-%s-token-application:%s' % (something, application.name)})

        if application.group_id:
            grants.update({'has-cred-%s-token-application-group:%s' % (something, application.group_id)})

    return grants


def _load_applications(tokens):
    """
    Заполняет атрибут application в данных токенах
    """
    apps, unknown_app_ids = providers.get_many_applications_by_ids(set(t.application_id for t in tokens))

    good_tokens = list()
    appless_tokens = list()
    for token in tokens:
        if token.application_id in unknown_app_ids:
            appless_tokens.append(token)
        else:
            good_tokens.append(token)

    if appless_tokens:
        formatted_token_ids = ', '.join([str(t.token_id) for t in appless_tokens])
        formatted_unknown_app_ids = ', '.join(map(str, unknown_app_ids))
        logger.debug(
            'Unknown application tokens found: tokens=%s, applications=%s' %
            (formatted_token_ids, formatted_unknown_app_ids),
        )

    app_id_to_app = {a.identifier: a for a in apps}

    tokens = list()
    for token in good_tokens:
        token.application = app_id_to_app[token.application_id]

    return good_tokens, appless_tokens
