# -*- coding: utf-8 -*-

import logging
import re
import socket
from socket import (
    AF_INET6,
    error as SocketError,
    getaddrinfo,
)
import ssl
from urlparse import (
    urlparse,
    urlunparse,
)
from uuid import uuid4
import warnings

from gevent.lock import RLock
from netaddr import (
    valid_ipv4,
    valid_ipv6,
)
import OpenSSL
from passport.backend.core.lazy_loader import LazyLoader
from passport.backend.core.logging_utils.helpers import mask_sensitive_fields
from passport.backend.core.logging_utils.loggers.graphite import GraphiteLogger
from passport.backend.core.utils.decorators import cached_property
from passport.backend.social.common._urllib3 import (
    PoolManager,
    ProxyManager,
    urllib3,
)
from passport.backend.social.common.cache import LocalMemoryCache
from passport.backend.social.common.chrono import now
from passport.backend.social.common.context import request_ctx
from passport.backend.social.common.exception import (
    OfflineZoraUseragentNotImplementedError,
    SocialException,
)
from passport.backend.social.common.misc import (
    trim_message,
    urlencode,
    urllib_quote,
    urlparse_qsl,
)
from passport.backend.social.common.social_config import social_config
from passport.backend.utils.common import (
    identity,
    remove_none_values,
    smart_text,
)
import yenv


USER_AGENT = 'yandex-social-useragent/0.1'

# О возможных значениях можно прочесть в man 3 SSL_get_version
FORBIDDEN_SSL_VERSIONS = {'SSLv2', 'SSLv3'}

# Каноничные имена статусов GoZora можно брать из
# https://wiki.yandex-team.ru/zora/gozora/instrukcija-polzovatelja/#kodyoshibok
ZORA_SSL_CERT_ERROR = 1000
# Эти коды я придумал сам. Нужно будет поменять, когда в GoZora появится свой
ZORA_INTERNAL_ERROR = 10000
ZORA_DNS_FAILURE = 10001

FAIL_REQUEST_RESPONSE = 'fail'
TIMEOUT_REQUEST_RESPONSE = 'timeout'
POOL_TIMEOUT_RESPONSE = 'pool_timeout'
SUCCESS_REQUEST_RESPONSE = 'success'
NETWORK_REQUEST_ERROR_CODE = 'network'
SERVICE_TEMPORARY_FAIL_REQUEST_ERROR_CODE = 'service_temporary_fail'

logger = logging.getLogger(__name__)
default_graphite_logger = GraphiteLogger(logging.getLogger('graphite.useragent'))


def build_http_pool_manager():
    ipv6_only = yenv._load('network', '') == 'ipv6only'

    return PoolManager(
        num_pools=social_config.connection_pool_limit,
        maxsize=social_config.connection_pool_maxsize,
        block=True,
        ssl_version=ssl.PROTOCOL_SSLv23,
        forbidden_ssl_versions=frozenset(FORBIDDEN_SSL_VERSIONS),

        cert_reqs=ssl.CERT_REQUIRED,
        ca_certs=social_config.ssl_certs_path,

        dns=_Dns(ipv6_only=ipv6_only),
    )


def get_http_pool_manager():
    return LazyLoader.get_instance('http_pool_manager')


def _build_zora_proxy_manager():
    _disable_zora_ssl_warnings()

    ipv6_only = yenv._load('network', '') == 'ipv6only'

    return ProxyManager(
        proxy_url=social_config.zora_url,
        num_pools=social_config.connection_pool_limit,
        maxsize=social_config.connection_pool_maxsize,
        block=True,

        # Сейчас подключиться к Zora можно только по http, но чтобы в будущем
        # для включения https было достаточно изменить схему в zora_url, здесь
        # мы подготовим параметры и для https подключений.
        ssl_version=ssl.PROTOCOL_SSLv23,
        forbidden_ssl_versions=frozenset(FORBIDDEN_SSL_VERSIONS),
        cert_reqs=ssl.CERT_NONE,
        ca_certs=social_config.ssl_certs_path,

        dns=_Dns(ipv6_only=ipv6_only),
    )


def _disable_zora_ssl_warnings():
    """
    Отключает варнинги о том что Gozora использует свои корневые сертификаты
    """
    zora_hostname = Url(social_config.zora_url).hostname
    warnings.filterwarnings(
        'ignore',
        r"Unverified HTTPS request is being made to host '%s'" % re.escape(zora_hostname),
        urllib3.exceptions.InsecureRequestWarning,
    )


def get_zora_useragent():
    return LazyLoader.get_instance('zora_useragent')


def _is_zora_dns_failure_exception(exc):
    if (
        isinstance(exc, urllib3.exceptions.ProxyError) and
        len(exc.args) >= 2 and
        isinstance(exc.args[1], socket.error)
    ):
        sock_exc = exc.args[1]
        if (
            len(sock_exc.args) >= 1 and
            sock_exc.args[0] == 'Tunnel connection failed: 502 resolving failed'
        ):
            return True


class UseragentError(SocialException):
    pass


class RequestError(UseragentError):
    pass


class NetworkFailureUseragentError(RequestError):
    def __init__(self, description, url, duration):
        super(NetworkFailureUseragentError, self).__init__(description)
        self.duration = duration
        self.url = url


class TimeoutUseragentError(RequestError):
    def __init__(self, description, url, duration):
        super(TimeoutUseragentError, self).__init__(description)
        self.duration = duration
        self.url = url


class ServiceTemporaryUnavailableUseragentError(UseragentError):
    def __init__(self, description, url, http_status):
        super(ServiceTemporaryUnavailableUseragentError, self).__init__(description)
        self.description = description
        self.url = url
        self.http_status = http_status


class ParsingUseragentError(UseragentError):
    def __init__(self, url, http_status):
        super(ParsingUseragentError, self).__init__()
        self.url = url
        self.http_status = http_status


class _DummyRequestSigner(object):
    def get_signed_request(self, request):
        return request


class ZoraError(SocialException):
    def __init__(self, error, description):
        message = 'Zora failed: %d %s' % (error, description)
        super(ZoraError, self).__init__(message)
        self.error = error
        self.description = description


class _BaseUseragent(object):
    request_error_class = RequestError
    service_temporary_unavailable_error_class = ServiceTemporaryUnavailableUseragentError
    network_exceptions = tuple()
    timeout_exceptions = tuple()

    timeout = None
    retries = None
    http = None

    def request(self, method, url, headers=None, fields=None, params=None,
                data=None, timeout=None, retries=None, request_signer=None,
                graphite_logger=None, statbox_logger=None, verify=None,
                parser=None, service_temporary_unavailable_exceptions=None,
                service=None, reconnect=None):
        parser = parser or identity

        if not service_temporary_unavailable_exceptions:
            service_temporary_unavailable_exceptions = tuple()
        elif not isinstance(service_temporary_unavailable_exceptions, tuple):
            service_temporary_unavailable_exceptions = tuple(service_temporary_unavailable_exceptions)

        if not graphite_logger:
            graphite_logger = default_graphite_logger

        unsigned_request = self._build_request(
            method,
            url,
            headers,
            params,
            data,
            fields,
            timeout,
            retries,
        )

        request_signer = request_signer or _DummyRequestSigner()
        request = self._sign_request(request_signer, unsigned_request)

        for i in xrange(request.retries):
            logger.debug(
                '%s requesting %s. Attempt number %d of %d' % (
                    type(self).__name__,
                    request,
                    i + 1,
                    request.retries,
                )
            )

            start_time = now.f()
            response = None

            try:
                response = self._try_request(request)
            except urllib3.exceptions.EmptyPoolError as exception:
                duration = now.f() - start_time
                self._log_empty_pool_fail(
                    request,
                    i,
                    duration,
                    service,
                    graphite_logger,
                    termination=True,
                )
                raise RequestError(unicode(exception))
            except (self.network_exceptions + self.timeout_exceptions) as exception:
                duration = now.f() - start_time
                self._log_temporary_fail(
                    request,
                    i,
                    duration,
                    exception,
                    response,
                    service,
                    graphite_logger,
                    termination=False,
                )

                error_args = (unicode(exception), url, duration)
                if isinstance(exception, self.timeout_exceptions):
                    error = TimeoutUseragentError(*error_args)
                elif isinstance(exception, self.network_exceptions):
                    error = NetworkFailureUseragentError(*error_args)

                if isinstance(exception, _RedirectError) and request.method in {'GET', 'HEAD'}:
                    unsigned_request = Request.from_request(unsigned_request, url=exception.location)
                    request = self._sign_request(request_signer, unsigned_request)
                continue

            duration = now.f() - start_time

            try:
                response.parsed = parser(response)
            except service_temporary_unavailable_exceptions as exception:
                self._log_temporary_fail(
                    request,
                    i,
                    duration,
                    exception,
                    response,
                    service,
                    graphite_logger,
                    termination=False,
                )
                error = ServiceTemporaryUnavailableUseragentError(
                    unicode(exception),
                    url,
                    response.status_code,
                )
                error.service_temporary_unavailable_exception = exception
                continue
            except Exception as e:
                self._log_permanent_fail(
                    request,
                    i,
                    duration,
                    response,
                    service,
                    graphite_logger,
                )

                parsing_error = ParsingUseragentError(url, response.status_code)
                parsing_error.cause = getattr(e, 'cause', None)
                e.cause = parsing_error
                raise

            self._log_success_request(
                request,
                i,
                duration,
                response,
                service,
                graphite_logger,
                termination=True,
            )

            return response

        raise error

    def _log_temporary_fail(
        self,
        request,
        retry_no,
        duration,
        exception,
        response,
        service,
        graphite_logger,
        termination,
    ):
        log_record = list()
        log_record.append(
            'Temporary fail during request: %s: Attempt number: %d of %d; '
            'Time elapsed: %.3f seconds' % (
                request,
                retry_no + 1,
                request.retries,
                duration,
            ),
        )
        if exception:
            log_record.append('Exception: %s(%s)' % (type(exception).__name__, str(exception)))
        if response:
            log_record.append(response.format(trim=True))
        logger.warning('; '.join(log_record))

        log_record = self._build_common_graphite_record(
            request,
            retry_no,
            duration,
            response,
            service,
            termination,
        )
        if isinstance(exception, self.timeout_exceptions):
            log_record.update(response=TIMEOUT_REQUEST_RESPONSE)
        else:
            log_record.update(response=FAIL_REQUEST_RESPONSE)
            if isinstance(exception, self.network_exceptions):
                log_record['error_code'] = NETWORK_REQUEST_ERROR_CODE
            else:
                log_record['error_code'] = SERVICE_TEMPORARY_FAIL_REQUEST_ERROR_CODE
        graphite_logger.log(**log_record)

    def _log_permanent_fail(self, request, retry_no, duration, response, service,
                            graphite_logger):
        logger.info(
            'Permanent fail during request: %s: Attempt number: %d of %d; '
            'Time elapsed: %.3f seconds; Response: %s' % (
                request,
                retry_no + 1,
                request.retries,
                duration,
                response.format(trim=True),
            ),
        )

    def _log_empty_pool_fail(
        self,
        request,
        retry_no,
        duration,
        service,
        graphite_logger,
        termination,
    ):
        logger.warning(
            'Useragent connection pool exhausted during request: %s: Attempt number: %d of %d; '
            'Time elapsed: %.3f seconds' % (
                request,
                retry_no + 1,
                request.retries,
                duration,
            ),
        )

        log_record = self._build_common_graphite_record(
            request,
            retry_no,
            duration,
            response=None,
            service=service,
            termination=termination,
        )
        log_record.update(response=POOL_TIMEOUT_RESPONSE)
        graphite_logger.log(**log_record)

    def _log_success_request(
        self,
        request,
        retry_no,
        duration,
        response,
        service,
        graphite_logger,
        termination,
    ):
        logger.info(
            'Success: %s. Attempt number: %d of %d; Time elapsed: %.3f seconds;' % (
                request,
                retry_no + 1,
                request.retries,
                duration,
            ),
        )

        log_record = self._build_common_graphite_record(
            request,
            retry_no,
            duration,
            response,
            service,
            termination,
        )
        log_record.update(response=SUCCESS_REQUEST_RESPONSE)
        graphite_logger.log(**log_record)

    def get(self, *args, **kwargs):
        return self.request('get', *args, **kwargs)

    def post(self, *args, **kwargs):
        return self.request('post', *args, **kwargs)

    def head(self, *args, **kwargs):
        return self.request('head', *args, **kwargs)

    def put(self, *args, **kwargs):
        return self.request('put', *args, **kwargs)

    def delete(self, *args, **kwargs):
        return self.request('delete', *args, **kwargs)

    def _build_request(self, method, url, headers, params, data, fields, timeout,
                       retries):
        if retries is None:
            retries = self.retries

        if timeout is None:
            timeout = self.timeout

        headers = dict(headers) if headers else dict()
        headers.setdefault('User-Agent', USER_AGENT)

        return Request(
            method=method,
            url=url,
            headers=headers,
            params=params,
            data=data,
            fields=fields,
            timeout=timeout,
            retries=retries,
        )

    def _sign_request(self, request_signer, unsigned_request):
        signed_request = request_signer.get_signed_request(unsigned_request)
        return self._rebuild_request_after_signing(signed_request)

    def _rebuild_request_after_signing(self, request):
        return request

    def _build_common_graphite_record(
        self,
        request,
        retry_no,
        duration,
        response,
        service,
        termination,
    ):
        log_record = dict(
            srv_hostname=request.hostname,
            duration=duration,
            request_id=request_ctx.request_id,
            tskv_format='social-useragent-log',
        )

        if response:
            log_record['http_code'] = response.status_code
        else:
            log_record['http_code'] = None

        if termination:
            retries_left = 0
        else:
            retries_left = request.retries - retry_no - 1

        if not self._in_passport_builder:
            log_record.update(
                service=service,
                retries_left=retries_left,
            )

        return log_record


class UserAgent(_BaseUseragent):
    """
    Высокоуровневый http агент, построенный поверх библиотеки urllib3
    """
    network_exceptions = (
        SocketError,
        urllib3.exceptions.HTTPError,
        OpenSSL.SSL.SysCallError,
    )
    timeout_exceptions = (
        urllib3.exceptions.TimeoutError,
    )

    def __init__(self, timeout=None, retries=None, pool_manager=None,
                 in_passport_builder=False):
        self.retries = retries or social_config.useragent_default_retries
        self.timeout = timeout or social_config.useragent_default_timeout
        self.http = pool_manager or build_http_pool_manager()
        self._in_passport_builder = in_passport_builder

    def _try_request(self, request):
        start_time = now.f()

        response = self.http.urlopen(
            method=request.method,
            url=request.url,
            headers=request.headers,
            body=request.data,
            timeout=request.timeout,
            pool_timeout=social_config.connection_pool_timeout,
            # retries не должны передаваться на более низкий уровень
            retries=False,
            redirect=False,
        )
        response.duration = now.f() - start_time

        if response.status // 100 == 3:
            raise _RedirectError(response.getheader('location'))

        return Response(
            status_code=response.status,
            content=response.data,
            duration=response.duration,
            headers=response.headers,
        )


class _ZoraUseragent(_BaseUseragent):
    network_exceptions = (
        SocketError,
        urllib3.exceptions.HTTPError,
        ZoraError,
    )
    timeout_exceptions = (
        urllib3.exceptions.TimeoutError,
    )

    def __init__(
        self,
        tvm_credentials_manager,
        timeout=None,
        retries=None,
        zora_connection_pool=None,
        zora_tvm_client_alias=None,
    ):
        self.retries = retries or social_config.useragent_default_retries
        self.timeout = timeout or social_config.useragent_default_timeout
        self.http = zora_connection_pool or _build_zora_proxy_manager()
        self._zora_tvm_client_alias = zora_tvm_client_alias or social_config.zora_tvm_client_alias
        self._tvm_credentials_manager = tvm_credentials_manager
        self._in_passport_builder = False

    def _try_request(self, request):
        start_time = now.f()

        self.http.proxy_headers.update({'X-Ya-Service-Ticket': self._get_tvm_service_ticket()})

        try:
            response = self.http.urlopen(
                method=request.method,
                url=request.url,
                headers=request.headers,
                body=request.data,
                timeout=request.timeout,
                pool_timeout=social_config.connection_pool_timeout,
                # retries не должны передаваться на более низкий уровень
                retries=False,
                redirect=False,
            )
        except urllib3.exceptions.ProxyError as e:
            if _is_zora_dns_failure_exception(e):
                raise ZoraError(ZORA_DNS_FAILURE, 'Dns failure')
            raise

        response.duration = now.f() - start_time

        try:
            proxy_status = int(response.getheader('X-Yandex-GoZora-Error-Code', '0'))
        except (TypeError, ValueError):
            logger.error(
                'Unexpected Zora response: status=%s, headers=%s, data=%s' %
                (response.status, response.headers, response.data),
            )
            raise ZoraError(ZORA_INTERNAL_ERROR, 'Unexpected Zora response')
        if proxy_status != 0:
            raise ZoraError(proxy_status, response.getheader('X-Yandex-Gozora-Error-Description', ''))

        if response.status // 100 == 3:
            raise _RedirectError(response.getheader('location'))

        return Response(
            status_code=response.status,
            content=response.data,
            duration=response.duration,
            headers=response.headers,
        )

    def _get_tvm_service_ticket(self):
        self._tvm_credentials_manager.load()
        return self._tvm_credentials_manager.get_ticket_by_alias(self._zora_tvm_client_alias)

    def _build_request(self, *args, **kwargs):
        request = super(_ZoraUseragent, self)._build_request(*args, **kwargs)
        headers = dict(request.headers)
        headers.update(self._build_zora_client_id_header())
        return Request.from_request(request, headers=headers)

    def _build_zora_client_id_header(self):
        return {'X-Ya-Client-Id': social_config.zora_client_id}


class ZoraUseragent(_ZoraUseragent):
    def _build_request(self, *args, **kwargs):
        request = super(ZoraUseragent, self)._build_request(*args, **kwargs)
        headers = dict(request.headers)
        # Т.к. zora кеширует ответы по URL и хедерам, нам нужно защититься от
        # получения наших данных другими потребителями из кеша, поэтому подсолим
        # хедеры и этим отключим механизм кеширования.
        headers.update(self._build_zora_salt_header())
        return Request.from_request(request, headers=headers)

    def _build_zora_salt_header(self):
        salt = uuid4().hex
        return {'X-Yandex-Socialism-Salt': salt}

    def _try_request(self, request):
        # URL'ы ресурсов загружаемых через Zora публично доступны, поэтому во
        # избежании утечки секретов через query, запретим все запросы, кроме POST.
        if request.method != 'POST':
            raise NotImplementedError('%s requests through Zora are forbidden' % request.method)
        return super(ZoraUseragent, self)._try_request(request)


class OfflineZoraUseragent(ZoraUseragent):
    # Реализация ZoraUseragent для запросов из скриптов, т.е. когда изначальный
    # запрос делается не конечным пользователем.

    def __init__(self):
        pass

    def request(self, *args, **kwargs):
        # TODO Завести в Zora источник для offline-запросов
        raise OfflineZoraUseragentNotImplementedError()


class _Dns(object):
    def __init__(self, ipv6_only=False):
        self._cache = LocalMemoryCache()
        self._lock = RLock()
        self._ipv6_only = ipv6_only

    def resolve(self, domain, ipv6_only=None):
        if ipv6_only is None:
            ipv6_only = self._ipv6_only

        ip_list = self._cache.get(domain)
        if not ip_list:
            with self._lock:
                # Вторая проверка нужна для процессов, которые ожидали
                # другой процесс. В этом случае здесь кеш уже будет заполнен и
                # нет нужды делать запрос в DNS ещё раз.
                ip_list = self._cache.get(domain)
                if not ip_list:
                    ip_list = self._get_ip_list(domain, ipv6_only)
                    logger.debug('Resolved domain name %s to %s' % (domain, ip_list))
                    self._cache.set(domain, ip_list)
        return ip_list[0]

    def invalidate(self, domain, ip_address):
        with self._lock:
            ip_list = self._cache.get(domain, [])
            if ip_address in ip_list:
                ip_list.remove(ip_address)
                ip_list.append(ip_address)

    def _get_ip_list(self, domain_name, ipv6_only):
        if valid_ipv4(domain_name) or valid_ipv6(domain_name):
            return [domain_name]

        args = (domain_name, None,)
        if ipv6_only:
            args += (AF_INET6,)
        address_struct_list = getaddrinfo(*args)

        ip_list = set()
        for address_struct in address_struct_list:
            address = address_struct[4][0]
            ip_list.add(address)

        return list(ip_list)


class _RedirectError(urllib3.exceptions.HTTPError):
    def __init__(self, location):
        super(_RedirectError, self).__init__(location)
        self.location = location

    def __str__(self):
        return 'Redirect to %s' % self.location


class Request(object):
    def __init__(self,
                 method,
                 url,
                 headers=None,
                 params=None,
                 data=None,
                 fields=None,
                 timeout=None,
                 retries=None,
                 not_quotable_params=None):
        self._method = method.upper()
        self._headers = headers or dict()

        self._url = Url(url, not_quotable_params=not_quotable_params)
        self._url.add_params(params)

        if data is not None:
            if isinstance(data, basestring):
                self._data = urlparse_qsl(data, keep_blank_values=True)
            else:
                self._data = data
        else:
            self._data = dict()

        if self._method_like_get(method):
            self._url.add_params(fields)
        elif self._method_like_post(method):
            self._data = self._add_params(self._data, fields)
        else:
            assert False, 'Unknown method'

        self._data = remove_none_values(self._data)

        self._timeout = timeout
        self._retries = 1 if retries is None else int(retries)

        if not_quotable_params is not None:
            self._not_quotable_params = set(not_quotable_params)
        else:
            self._not_quotable_params = set()

    def __str__(self):
        if self._timeout is not None:
            timeout = '%.2f' % self.timeout
        else:
            timeout = 'None'

        if self._retries is not None:
            retries = '%d' % self.retries
        else:
            retries = 'None'

        return (
            'HTTP request to %(url)s, method=%(method)s, data=%(data)s, '
            'headers=%(headers)s, timeout=%(timeout)s, retries=%(retries)s' %
            dict(
                url=self.url_masked_for_log,
                method=self.method,
                data=self.data_masked_for_log,
                headers=self.headers_masked_for_log,
                timeout=timeout,
                retries=retries,
            )
        )

    def __repr__(self):
        return '<' + str(self) + '>'

    def __eq__(self, other):
        if type(other) is not Request:
            raise NotImplementedError(other)  # pragma: no cover
        return self.to_dict() == other.to_dict()

    def __ne__(self, other):
        return not self.__eq__(other)

    def _method_like_get(self, method):
        return method.upper() in {'GET', 'HEAD', 'DELETE'}

    def _method_like_post(self, method):
        return method.upper() in {'POST', 'PUT', 'PATCH'}

    def _add_params(self, fst, sec):
        fst = fst or []
        sec = sec or []
        fst = self._dict_to_list(fst)
        sec = self._dict_to_list(sec)
        return fst + sec

    def _dict_to_list(self, value):
        if isinstance(value, dict):
            value = value.items()
        assert isinstance(value, list), repr(value)
        return value

    @classmethod
    def from_request(cls, request, url=None, headers=None, data=None, not_quotable_params=None):
        return cls(
            method=request.method,
            url=url or request.url,
            headers=headers or request.headers,
            data=data or request.data,
            timeout=request.timeout,
            retries=request.retries,
            not_quotable_params=not_quotable_params or request._not_quotable_params,
        )

    @property
    def url(self):
        return str(self._url)

    @property
    def url_masked_for_log(self):
        return self._url.masked_for_log

    @property
    def data(self):
        if self._data:
            return urlencode(self._data)

    @property
    def data_masked_for_log(self):
        if self._data:
            return urlencode(mask_sensitive_fields(dict(self._data)))

    @property
    def headers(self):
        headers = dict(self._headers)
        if self._method_like_post(self._method):
            headers.update({'Content-Type': 'application/x-www-form-urlencoded'})
        return headers

    @property
    def headers_masked_for_log(self):
        return mask_sensitive_fields(self.headers)

    @property
    def method(self):
        return self._method

    @property
    def hostname(self):
        return self._url.hostname

    @property
    def retries(self):
        return self._retries

    @property
    def timeout(self):
        return self._timeout

    def to_dict(self):
        return dict(
            method=self._method,
            url=self._url.to_dict(),
            headers=self.headers,
            data=self.data,
            timeout=self.timeout,
            retries=self.retries,
            not_quotable_params=self._not_quotable_params,
        )


class Response(object):
    def __init__(self, status_code, content, duration, headers=None, parsed=None):
        self.status_code = self.status = status_code
        self.content = self.data = content or ''
        self.duration = duration
        self.parsed = parsed

        headers = headers or dict()
        self.headers = {k.lower(): v for k, v in headers.iteritems()}

    def getheader(self, key, default=None):
        return self.headers.get(key.lower(), default)

    def format(self, trim=False):
        bits = [
            u'status=' + str(self.status),
        ]
        if self.content:
            bits.append(u'content=' + smart_text(self.content, errors='replace'))
        text = u' '.join(bits)
        if trim:
            text = trim_message(text)
        return u'Response(%s)' % text

    @cached_property
    def decoded_data(self):
        if self.content:
            return self.content.decode('utf-8')
        else:
            return ''


class Url(object):
    def __init__(self, url, params=None, not_quotable_params=None):
        self._url, self._params = self._split_url_and_query(url)
        self._params = self._add_params(self._params, params)

        self._params = remove_none_values(self._params)

        if not_quotable_params is not None:
            self._not_quotable_params = set(not_quotable_params)
        else:
            self._not_quotable_params = set()

    def _split_url_and_query(self, url):
        url = urlparse(url)

        # urlparse_qsl раскворитует параметры, так что дополнительно это делать
        # не нужно.
        params = urlparse_qsl(url.query, keep_blank_values=True)

        url = list(url)
        url[4] = None
        return url, params

    def _add_params(self, fst, sec):
        fst = fst or []
        sec = sec or []
        fst = self._dict_to_list(fst)
        sec = self._dict_to_list(sec)
        return fst + sec

    def _dict_to_list(self, value):
        if isinstance(value, dict):
            value = value.items()
        assert isinstance(value, list), repr(value)
        return value

    def _get_quoted_path(self, path):
        if isinstance(path, unicode):
            path = path.encode('utf-8')
        return urllib_quote(path, safe='/%@')

    def _get_quoted_params(self, params):
        quoted_params = []
        for key, value in params:
            if key not in self._not_quotable_params:
                key = self._url_quote_param(key)
                value = self._url_quote_param(value)
            quoted_params.append((key, value))
        return quoted_params

    def _url_quote_param(self, s):
        return urllib_quote(s, safe='')

    def __str__(self):
        url = list(self._url)
        url[2] = self._get_quoted_path(url[2])
        url[4] = self.query
        return urlunparse(url)

    @property
    def masked_for_log(self):
        url = list(self._url)
        url[2] = self._get_quoted_path(url[2])
        masked_params = mask_sensitive_fields(dict(self._params)).items()
        url[4] = self.build_query(masked_params)
        return urlunparse(url)

    @property
    def query(self):
        return self.build_query(self._params)

    def build_query(self, params):
        quoted_params = self._get_quoted_params(params)
        rels = [k + '=' + v for k, v in quoted_params]
        return '&'.join(rels)

    def add_params(self, params):
        params = params or []
        params = remove_none_values(params)
        self._params = self._add_params(self._params, params)
        return self

    def remove_params(self, keys):
        self._params = [(k, v) for k, v in self._params if k not in keys]

    def _get_scheme(self):
        return self._url[0]

    def _set_scheme(self, value):
        self._url[0] = value

    scheme = property(_get_scheme, _set_scheme)

    @property
    def hostname(self):
        parsed_url = urlparse(self.paramless)
        return parsed_url.hostname

    def to_dict(self):
        return {
            'url': urlunparse(self._url),
            # Считаем, что порядок параметров не важен
            'params': dict(self._params),
            'not_quotable_params': self._not_quotable_params,
        }

    @property
    def paramless(self):
        return urlunparse(self._url)

    @property
    def params(self):
        return dict(self._params)

    @property
    def path(self):
        return self._url[2]
