# -*- coding: utf-8 -*-

import re
import unicodedata
from urlparse import urlsplit

from formencode import Invalid
from formencode.validators import (
    _,
    URL,
)
from passport.backend.core.validators import (
    ASCIIString,
    Int,
    Name,
    Number,
    Schema,
    SimpleEmailValidator,
    String,
    StringBool,
    Unixtime,
)
from passport.backend.core.validators.utils import convert_formencode_invalid_to_error_list
from passport.backend.social.common.fraud import find_url
from passport.backend.social.common.misc import split_scope_string


class Token(String):
    not_empty = True
    max = 30000


class ClientId(ASCIIString):
    max = 100
    not_empty = True
    strip = True


class ClientSecret(ASCIIString):
    max = 100
    not_empty = True
    strip = True


class Consumer(String):
    not_empty = True


class ProviderCode(String):
    not_empty = True


class Userid(String):
    not_empty = True


class ConsumerForm(Schema):
    consumer = Consumer()


class SessionId(String):
    not_empty = True


class ApplicationName(ASCIIString):
    max = 160
    not_empty = True
    strip = True


class Username(String):
    not_empty = True
    strip = True
    max = 255


class Email(SimpleEmailValidator):
    not_empty = True


class Scope(ASCIIString):
    strip = True
    not_empty = True

    messages = {
        'invalid': _('Invalid value'),
    }

    def _to_python(self, value, state):
        scopes = split_scope_string(value)
        if not scopes and (self.not_empty or value):
            raise Invalid(self.message('invalid', state), value, state)

        return scopes


class ScopeString(ASCIIString):
    max = 500
    not_empty = True
    strip = True


class CsvSet(String):
    strip = True
    not_empty = False
    if_missing = if_empty = frozenset()
    value_validator = String

    messages = {
        'badValues': _('The values %(fields)s are not valid'),
    }

    def _to_python(self, value, state):
        raw_values = [i.strip() for i in value.split(',')]

        valid_values = set()
        invalid_values = set()
        value_validator = self.value_validator()
        for raw_value in raw_values:
            try:
                valid_value = value_validator.to_python(raw_value)
                valid_values.add(valid_value)
            except Invalid:
                invalid_values.add(raw_value)

        if invalid_values:
            formatted_values = ', '.join(sorted(map(str, invalid_values)))
            raise Invalid(self.message('badValues', state, fields=formatted_values), value, state)

        return frozenset(valid_values)


class ProfileId(Int):
    not_empty = True


class ProfileIdSet(CsvSet):
    value_validator = ProfileId


class ApplicationNameSet(CsvSet):
    value_validator = ApplicationName


class _BaseUrl(URL):
    add_http = False
    check_exists = False
    not_empty = True
    strip = True
    allow_fragments = True

    messages = {
        'schemeNotAllowed': _('Scheme is not allowed'),
        'credentialsNotAllowed': _('Credentials are not allowed'),
        'fragmentNotAllowed': _('Fragment is not allowed'),
        'tooLong': _('Too long'),
    }

    def _to_python(self, value, state):
        url = super(_BaseUrl, self)._to_python(value, state)

        if self.max is not None and len(url) > self.max:
            raise Invalid(self.message('tooLong', state), value, state)

        url_components = urlsplit(url)
        if url_components.scheme != 'https':
            raise Invalid(self.message('schemeNotAllowed', state), value, state)
        if url_components.username is not None or url_components.password is not None:
            raise Invalid(self.message('credentialsNotAllowed', state), value, state)
        if not self.allow_fragments and url_components.fragment:
            raise Invalid(self.message('fragmentNotAllowed', state), value, state)
        return url


class AuthorizationUrl(_BaseUrl):
    max = 200
    not_empty = True


class TokenUrl(_BaseUrl):
    max = 200
    not_empty = True
    allow_fragments = False


class Gpt(String):
    not_empty = True


class ApplicationDisplayName(String):
    max = 100
    not_empty = True
    strip = True
    allowed_unicode_general_categories = {
        'Lu',
        'Ll',
        'Nd',
        'Pc',
        'Pd',
        'Ps',
        'Pe',
        'Pi',
        'Pf',
        'Po',
        'Zs',
        'Sm',
    }
    tag_re = re.compile(r'(<.+?>)')

    messages = {
        'htmlNotAllowed': _('HyperText not allowed: %(word)s'),
    }

    def __init__(self, *args, **kw):
        super(ApplicationDisplayName, self).__init__(*args, **kw)
        self.printable_letters_validator = UnicodeGeneralCategoryValidator(
            allowed_unicode_general_categories=self.allowed_unicode_general_categories,
        )
        self.url_blocker_validator = UrlBlockerValidator(allow_netloc=True)

    def validate_python(self, value, state):
        super(ApplicationDisplayName, self).validate_python(value, state)
        self.printable_letters_validator.validate_python(value, state)
        self.url_blocker_validator.validate_python(value, state)
        self._block_html_tags(value, state)

    def _block_html_tags(self, value, state):
        match = self.tag_re.search(value)
        if match:
            raise Invalid(
                self.message('htmlNotAllowed', state, word=match.group(1)),
                value,
                state,
            )


class UnicodeGeneralCategoryValidator(String):
    """
    Проверяет, что строка состоит только из допустимых символов.

    Допустимые символы задаются Unicode-ными "общими категориями":
    http://www.unicode.org/reports/tr44/tr44-4.html#General_Category_Values
    """

    allowed_unicode_general_categories = set()

    messages = {
        'invalid': _('Letter %(letter)s not allowed'),
    }

    def validate_python(self, value, state):
        super(UnicodeGeneralCategoryValidator, self).validate_python(value, state)
        for letter in value:
            if unicodedata.category(letter) not in self.allowed_unicode_general_categories:
                raise Invalid(
                    self.message('invalid', state, letter=letter),
                    value,
                    state,
                )


class UrlBlockerValidator(String):
    """
    Проверяет, что в строке нет URL'ов.
    """

    allow_netloc = False

    messages = {
        'invalid': _('Urls not allowed'),
    }

    def validate_python(self, value, state):
        super(UrlBlockerValidator, self).validate_python(value, state)
        if find_url(
            value,
            find_netloc_enabled=not self.allow_netloc,
        ):
            raise Invalid(
                self.message('invalid', state),
                value,
                state,
            )


class DeviceInfoForm(Schema):
    am_version_name = String(if_missing=None, strip=True)
    app_id = String(if_missing=None, strip=True)
    app_platform = String(if_missing=None, strip=True)
    app_version = String(if_missing=None, strip=True)
    manufacturer = String(if_missing=None, strip=True)
    model = String(if_missing=None, strip=True)
    uuid = String(if_missing=None, strip=True)
    device_name = String(if_missing=None, strip=True)
    os_version = String(if_missing=None, strip=True)
    # Идентификатор устройства для iPhone'а передаётся из АМа
    ifv = String(if_missing=None, strip=True)
    # Идентификатор устройства для Android'а передаётся из АМа
    deviceid = String(if_missing=None, strip=True)
    # Идентификатор устройства
    device_id = String(if_missing=None, strip=True)

    DEVICE_INFO_FIELD_NAMES = (
        'am_version_name',
        'app_id',
        'app_platform',
        'manufacturer',
        'model',
        'app_version',
        'uuid',
        'deviceid',
        'ifv',
        'device_name',
        'os_version',
        'device_id',
    )


class TaskId(ASCIIString):
    id_re = re.compile('^[a-z0-9]{5,32}$', re.IGNORECASE)
    messages = {
        'invalid': _('Invalid value'),
    }
    max = 64
    not_empty = True
    separator_re = re.compile('[-:]')
    strip = True

    def _to_python(self, value, state):
        return self.separator_re.sub('', value)

    def validate_python(self, value, state):
        if not self.id_re.match(value):
            raise Invalid(self.message('invalid'), value, state)


__all__ = [
    'convert_formencode_invalid_to_error_list',
    'Name',
    'Number',
    'StringBool',
    'Unixtime',
]
