# -*- coding: utf-8 -*-

from __future__ import unicode_literals

import logging

from flask import (
    Request as _Request,
    Response as _Response,
)
from passport.backend.core.exceptions import UnknownUid
from passport.backend.core.models.account import Account
from passport.backend.core.tvm.tvm_credentials_manager import get_tvm_credentials_manager
from passport.backend.core.types.ip.ip import IP
from passport.backend.core.utils.blackbox import get_many_accounts_by_uids
from passport.backend.core.utils.decorators import cached_property
from passport.backend.social.common import exception as social_exceptions
from passport.backend.social.common.builders.blackbox import (
    add_phone_arguments,
    Blackbox,
    BlackboxInvalidResponseError,
    BlackboxInvalidSessionidError,
    BlackboxOauthTokenInvalidError,
    BlackboxTemporaryError,
    BlackboxUnknownError,
    check_oauth_response,
    check_oauth_response_suitable_for_binding,
    check_session_cookie,
)
from passport.backend.social.common.builders.kolmogor import Kolmogor
from passport.backend.social.common.builders.passport import Passport
from passport.backend.social.common.context import request_ctx
from passport.backend.social.common.error_handler import ErrorHandler as _ErrorHandler
from passport.backend.social.common.exception import (
    DatabaseFailureSourceTypeMixin,
    ExternalFailureSourceTypeMixin,
    NetworkFailureSourceTypeMixin,
    NotErrorFailureSourceTypeMixin,
)
from passport.backend.social.common.grants import (
    check_all_of_grants,
    GrantsContext,
    GrantsMissingError as CommonGrantsMissingError,
)
from passport.backend.social.common.misc import (
    build_request_id,
    dump_to_json_string,
    trim_message,
)
from passport.backend.social.common.redis_client import RedisError
from passport.backend.social.common.session import Session
from passport.backend.social.common.useragent import get_http_pool_manager
from passport.backend.social.common.validators import (
    ConsumerForm,
    convert_formencode_invalid_to_error_list,
    Invalid as ValidatorInvalid,
)
import ticket_parser2.exceptions
from werkzeug import BaseResponse as _WerkzeugBaseResponse
from werkzeug.exceptions import HTTPException as _WerkzeugHttpExcetpion
from werkzeug.test import EnvironBuilder


logger = logging.getLogger(__name__)


def description_for_consumer_from_grants_missing_error(error):
    """
    Строит описание отказа из-за нехватки грантов для потребителя.
    """
    if not error.failed_to_parse_ticket and error.missing_grants:
        comma_separated_missing_grants = ', '.join(sorted(error.missing_grants))
        description = 'Missing grants [%s] from %s' % (comma_separated_missing_grants, str(error.grants_context))
    elif error.failed_to_parse_ticket:
        description = 'Failed to parse ticket from %s' % str(error.grants_context)
    else:
        description = ''
    return description


def _save_response_dict_to_log(response_dict):
    formatted_dict = trim_message(str(response_dict), cut=False)
    logger.debug('Response sent: %s' % formatted_dict)


class InternalHandlerV2(object):
    basic_form = None
    required_grants = []
    _response_class = None

    def __init__(self, request):
        self._request = request
        self._consumer = None
        self.form_values = dict()
        self.response_values = dict()

    def get(self, *args, **kwargs):
        try:
            self._process_root_form()
            request_ctx.grants_context = self._get_grants_context()
            self._check_required_grants()
            self._process_basic_form()
            self._process_request()
        except Exception as e:
            codes, description = self._process_exception(e)
            response = self._response_fail(codes, description)
        else:
            response = self._response_success()
        return response

    def _process_request(self):
        raise NotImplementedError()  # pragma: no cover

    def _process_root_form(self):
        values = self._process_form(ConsumerForm())
        self._consumer = values['consumer']

    def _process_basic_form(self):
        if self.basic_form is not None:
            values = self._process_form(self.basic_form)
            self.form_values.update(values)

    def _process_form(self, form):
        try:
            retval = form.to_python(self._request_values, state=None)
        except ValidatorInvalid as e:
            validation_errors = convert_formencode_invalid_to_error_list(e)
            raise FormValidationWebServiceError(codes=validation_errors)
        return retval

    @property
    def _request_values(self):
        retval = dict()
        retval.update(self._request.args)
        retval.update(self._request.form)
        return retval

    @property
    def _user_ip(self):
        if not self._request.user_ip:
            raise UserIpMissingWebServiceError()
        return self._request.user_ip

    @property
    def _consumer_ip(self):
        if not self._request.consumer_ip:
            raise ConsumerIpMissingWebServiceError()
        return self._request.consumer_ip

    @property
    def _grants_context(self):
        return request_ctx.grants_context

    def _get_grants_context(self):
        return GrantsContext(
            consumer_ip=self._consumer_ip,
            consumer=self._consumer,
            ticket_body=self._request.ticket_body,
        )

    def _check_required_grants(self):
        grants_config = self._get_grants_config()
        grants_config.load()
        try:
            check_all_of_grants(grants_config, self._grants_context, self.required_grants)
        except CommonGrantsMissingError as e:
            logger.warning('Access denied (%s)' % str(e))
            raise GrantsMissingWebServiceError(e)
        logger.info('Access to %s is granted for %s' % (sorted(self.required_grants), str(self._grants_context)))

    def _get_grants_config(self):
        raise NotImplementedError()  # pragma: no cover

    def _build_json_response(self, value, http_status, api_status, api_error_code=None):
        response = self._response_class(status=http_status, mimetype='application/json')
        response.data = dump_to_json_string(value)
        response.api_status = api_status
        response.api_error_code = api_error_code
        return response

    def _build_file_response(self, _file, filename, mimetype, http_status):
        headers = {'Content-Disposition': 'attachment; filename="%s"' % filename}
        response = self._response_class(
            status=http_status,
            mimetype=mimetype,
            headers=headers,
            direct_passthrough=True,
        )
        response.data = _file.read()
        return response

    def _response_success(self):
        return self._build_json_response(
            dict(status='ok', **self.response_values),
            http_status=200,
            api_status='ok',
        )

    def _response_fail(self, codes, description):
        response = dict(status='error', errors=list(codes), request_id=self._request.id)
        if description:
            response.update(description=description)
        response = dict(response, **self.response_values)
        _save_response_dict_to_log(response)
        return self._build_json_response(
            response,
            http_status=200,
            api_status='error',
            api_error_code=codes[0],
        )

    def _process_exception(self, exception):
        logger.debug('Processing exception %s' % type(exception).__name__)
        error_handler = self._build_error_handler(exception)
        codes, description = error_handler.exception_to_response()
        error_handler.exception_to_graphite()
        return codes, description

    def _build_error_handler(self, exception):
        return ErrorHandler(exception, handler_id=type(self).__name__)


class ErrorHandler(_ErrorHandler):
    def __init__(self, exception, handler_id):
        super(ErrorHandler, self).__init__(exception)
        self._handler_id = handler_id

    def exception_to_response(self):
        if isinstance(self._exception, WebServiceError):
            codes, description = self._exception.codes, self._exception.description
        elif isinstance(
            self._exception,
            (
                BlackboxInvalidResponseError,
                BlackboxTemporaryError,
                BlackboxUnknownError,
            ),
        ):
            codes = BlackboxFailedWebServiceError().codes
            description = BlackboxFailedWebServiceError().description
        elif isinstance(self._exception, (social_exceptions.DatabaseError, RedisError)):
            codes = DatabaseFailedWebServiceError().codes
            description = DatabaseFailedWebServiceError().description
        elif isinstance(self._exception, social_exceptions.NetworkProxylibError):
            codes = NetworkFailedWebServiceError().codes
            description = NetworkFailedWebServiceError().description
        elif isinstance(self._exception, (
            social_exceptions.ProviderTemporaryUnavailableProxylibError,
            social_exceptions.UnexpectedResponseProxylibError,
        )):
            codes = ProviderFailedWebServiceError().codes
            description = ProviderFailedWebServiceError().description
        else:
            logger.error('Exception unhandled', exc_info=True)
            codes, description = ['exception.unhandled'], ''
        return codes, description

    def _get_handler_id(self):
        return self._handler_id


class ExternalHandler(object):
    basic_form = None
    _response_class = None

    def __init__(self, request):
        self._request = ExternalRequest(request.environ)
        self.form_values = dict()
        self.response_values = dict()

    def get(self, *args, **kwargs):
        try:
            self._process_basic_form()
            self._process_request()
        except Exception as e:
            codes, description = self._process_exception(e)
            response = self._response_fail(codes, description)
        else:
            response = self._response_success()
        return response

    def _process_request(self):
        raise NotImplementedError()  # pragma: no cover

    def _process_basic_form(self):
        if self.basic_form is not None:
            values = self._process_form(self.basic_form)
            self.form_values.update(values)

    def _process_form(self, form):
        try:
            retval = form.to_python(self._request_values, state=None)
        except ValidatorInvalid as e:
            validation_errors = convert_formencode_invalid_to_error_list(e)
            raise FormValidationWebServiceError(codes=validation_errors)
        return retval

    @property
    def _request_values(self):
        retval = dict()
        retval.update(self._request.args)
        retval.update(self._request.form)
        return retval

    @property
    def _user_ip(self):
        if not self._request.user_ip:
            raise UserIpMissingWebServiceError()
        return self._request.user_ip

    def _build_json_response(self, value, http_status, api_status, api_error_code=None):
        response = self._response_class(status=http_status, mimetype='application/json')
        response.data = dump_to_json_string(value)
        response.api_status = api_status
        response.api_error_code = api_error_code
        return response

    def _response_success(self):
        return self._build_json_response(
            dict(status='ok', **self.response_values),
            http_status=200,
            api_status='ok',
        )

    def _response_fail(self, codes, description):
        response = dict(status='error', errors=list(codes), request_id=self._request.id)
        if description:
            response.update(description=description)
        response = dict(response, **self.response_values)
        _save_response_dict_to_log(response)
        return self._build_json_response(
            response,
            http_status=200,
            api_status='error',
            api_error_code=codes[0],
        )

    def _process_exception(self, exception):
        logger.debug('Processing exception %s' % type(exception).__name__)
        error_handler = self._build_error_handler(exception)
        codes, description = error_handler.exception_to_response()
        error_handler.exception_to_graphite()
        return codes, description

    def _build_error_handler(self, exception):
        return ErrorHandler(exception, handler_id=type(self).__name__)


class Request(_Request):
    def __init__(self, environ):
        super(Request, self).__init__(environ)

        self.args = self.args.to_dict()
        self.form = self.form.to_dict()
        self.values = self.values.to_dict()

        self.id = self.headers.get('X-Request-Id') or build_request_id()
        self.yandex_authorization = self.headers.get('Ya-Consumer-Authorization')

        self.user_ip = self.headers.get('Ya-Consumer-Client-Ip')
        self.user_ip = self.user_ip and IP(self.user_ip)

        self.consumer_ip = self.headers.get('X-Real-Ip') or self.remote_addr
        self.consumer_ip = self.consumer_ip and IP(self.consumer_ip)

        self.header_consumer = self.headers.get('Ya-Consumerid')
        self.header_provider_token = self.headers.get('X-Social-Access-Token-Value')
        self.header_provider_token_secret = self.headers.get('X-Social-Access-Token-Secret')
        self.ticket_body = self.headers.get('X-Ya-Service-Ticket')
        self.user_ticket = self.headers.get('X-Ya-User-Ticket')

    @classmethod
    def create(cls, url=None, method=None, args=None, form=None, id=None,
               authorization=None, user_ip=None, consumer_ip=None,
               header_consumer=None, header_provider_token=None,
               header_provider_token_secret=None, ticket_body=None,
               user_ticket=None):
        kwargs = dict(
            query_string=args,
            data=form,
        )
        if method:
            kwargs.update(method=method)
        if url:
            kwargs.update(path=url)
        builder = EnvironBuilder(**kwargs)
        request = builder.get_request(cls)
        if id:
            request.id = id
        if authorization:
            request.authorization = authorization
        if user_ip:
            request.user_ip = IP(user_ip)
        if consumer_ip:
            request.consumer_ip = IP(consumer_ip)
        if header_consumer:
            request.header_consumer = header_consumer
        if header_provider_token:
            request.header_provider_token = header_provider_token
        if header_provider_token_secret:
            request.header_provider_token_secret = header_provider_token_secret
        if ticket_body:
            request.ticket_body = ticket_body
        if user_ticket:
            request.user_ticket = user_ticket
        return request


class ExternalRequest(_Request):
    def __init__(self, environ):
        super(ExternalRequest, self).__init__(environ)

        self.args = self.args.to_dict()
        self.form = self.form.to_dict()
        self.values = self.values.to_dict()

        self.id = self.headers.get('X-Request-Id') or build_request_id()

        self.user_ip = self.headers.get('X-Real-Ip') or self.remote_addr
        self.user_ip = self.user_ip and IP(self.user_ip)

        self.consumer_ip = self.user_ip


class Response(_Response):
    _werkzeug_response_constructor_args = [
        'content_type',
        'direct_passthrough',
        'headers',
        'mimetype',
        'response',
        'status',
    ]

    def __init__(self, *args, **kwargs):
        super(Response, self).__init__(*args, **kwargs)
        # api_status -- код успешности выполнения запроса (ok, error)
        self.api_status = None
        # api_error_code -- код отказа ручки
        self.api_error_code = None

    @classmethod
    def force_type(cls, response, environ=None):
        # Согласно документации response может быть
        #   * объектом werkzeug.BaseResponse
        #   * Wsgi-приложением
        # Опытным путём выяснил, что response также может быть объектом
        # werkzeug.HttpException.

        if isinstance(response, cls):
            return response

        if isinstance(response, _WerkzeugHttpExcetpion):
            response = response.get_response(environ)

        if isinstance(response, _WerkzeugBaseResponse):
            kwargs = dict()
            for kwarg_name in cls._werkzeug_response_constructor_args:
                if hasattr(response, kwarg_name):
                    kwargs[kwarg_name] = getattr(response, kwarg_name)
            return cls(**kwargs)

        # Что делать с Wsgi-приложением не очень понятно, т.к. ему нужно
        # передать start_response, который не ясно откуда брать.

        raise NotImplementedError()


class WebServiceError(Exception):
    codes = []
    code = None
    description = ''

    def __init__(self, codes=None, description=None):
        assert self.code or self.codes or codes
        if codes:
            self.codes = codes
        elif not self.codes and self.code:
            self.codes = [self.code]

        if description:
            self.description = description


class FormValidationWebServiceError(NotErrorFailureSourceTypeMixin, WebServiceError):
    pass


class GrantsMissingWebServiceError(NotErrorFailureSourceTypeMixin, WebServiceError):
    code = 'access.denied'

    def __init__(self, error):
        super(GrantsMissingWebServiceError, self).__init__()
        self.description = description_for_consumer_from_grants_missing_error(error)


class SocialInternalHandlerV2(InternalHandlerV2):
    @property
    def _account_getter(self):
        return AccountGetter(get_http_pool_manager(), self._request)

    @property
    def _blackbox(self):
        return Blackbox(get_http_pool_manager())

    @property
    def _kolmogor(self):
        return Kolmogor(get_http_pool_manager())

    @property
    def _passport_api(self):
        return Passport(get_http_pool_manager())

    def _get_account_from_token(self):
        token = self._get_token()
        return self._account_getter.get_account_from_token(token, self._user_ip)

    def _get_account_from_uid(self, uid, **kwargs):
        return self._account_getter.get_account_from_uid(uid, self._user_ip, **kwargs)

    def _get_many_accounts_from_uids(self, uids):
        return self._account_getter.get_many_accounts_from_uids(uids, self._user_ip)

    def _get_token(self):
        authorization = self._request.yandex_authorization or ''
        authorization = authorization.strip()
        token_prefix = 'Bearer '
        if not authorization.lower().startswith(token_prefix.lower()):
            raise YandexTokenInvalidWebServiceError()
        # Здесь не нужно проверять оставшуюся часть токена на пустоту, т.к.
        # этот случай уже покрывается проверкой выше.
        return authorization[len(token_prefix):]

    @cached_property
    def _session(self):
        return Session()

    def _get_account_from_user_ticket(self, required_scopes=None, uid=None):
        return self._account_getter.get_account_from_user_ticket(self._user_ip, required_scopes, uid)


class AccountGetter(object):
    def __init__(self, http_pool_manager, request):
        self._http_pool_manager = http_pool_manager
        self._request = request

    @property
    def _blackbox(self):
        return Blackbox(self._http_pool_manager)

    def _parse_account(self, blackbox_response):
        return Account().parse(blackbox_response)

    def get_account_from_token(self, token, user_ip):
        oauth_response = self._blackbox.oauth(
            **add_phone_arguments(
                oauth_token=token,
                ip=user_ip,
                dbfields=[],
                attributes=[],
            )
        )
        try:
            check_oauth_response(oauth_response)
            check_oauth_response_suitable_for_binding(oauth_response)
        except BlackboxOauthTokenInvalidError:
            raise YandexTokenInvalidWebServiceError()
        return self._parse_account(oauth_response)

    def get_account_from_uid(self, uid, user_ip):
        userinfo_response = self._blackbox.userinfo(
            **add_phone_arguments(
                uid=uid,
                ip=user_ip,
                dbfields=[],
                attributes=[],
            )
        )
        try:
            return self._parse_account(userinfo_response)
        except UnknownUid:
            raise AccountNotFoundWebServiceError()

    def get_many_accounts_from_uids(self, uids, user_ip=None):
        userinfo_args = add_phone_arguments(
            attributes=['account.is_disabled'],
            dbfields=[],
        )
        if user_ip:
            userinfo_args.update(ip=str(user_ip))
        return get_many_accounts_by_uids(
            uids=uids,
            blackbox_builder=self._blackbox,
            userinfo_args=userinfo_args,
        )

    def get_account_from_session_id(self, session_id, user_ip, host):
        sessionid_response = self._blackbox.sessionid(
            **add_phone_arguments(
                sessionid=session_id,
                ip=user_ip,
                host=host,
                dbfields=[],
                attributes=[],
            )
        )
        try:
            check_session_cookie(sessionid_response)
            return self._parse_account(sessionid_response)
        except BlackboxInvalidSessionidError:
            raise YandexSessionInvalidWebServiceError()

    def get_account_from_user_ticket(self, user_ip, required_scopes=None, uid=None):
        user_ticket = self.check_user_ticket(required_scopes)

        if uid is None:
            if not user_ticket.default_uid:
                raise TvmUserTicketNoUidWebServiceError(user_ticket.uids)
            uid = user_ticket.default_uid
        if uid not in user_ticket.uids:
            raise TvmUserTicketNoUidWebServiceError(user_ticket.uids)

        try:
            return self.get_account_from_uid(uid, user_ip)
        except AccountNotFoundWebServiceError:
            raise TvmUserTicketInvalidWebServiceError(description='Account not found')

    def check_user_ticket(self, required_scopes=None):
        if required_scopes and isinstance(required_scopes, basestring):
            required_scopes = [required_scopes]

        user_context = get_tvm_credentials_manager().get_user_context()
        if self._request.user_ticket is None:
            raise TvmUserTicketInvalidWebServiceError(description='No ticket')
        try:
            user_ticket = user_context.check(self._request.user_ticket)
        except ticket_parser2.exceptions.TicketParsingException as e:
            logger.debug('Invalid TVM user ticket: %s, %s, %s' % (e.status, e.message, e.debug_info))
            raise TvmUserTicketInvalidWebServiceError(description=e.message)

        if required_scopes and not any([user_ticket.has_scope(s) for s in required_scopes]):
            raise TvmUserTicketMissingScopesWebServiceError(user_ticket.scopes, required_scopes)

        return user_ticket


class UserIpMissingWebServiceError(NotErrorFailureSourceTypeMixin, WebServiceError):
    code = 'user_ip.empty'


class ConsumerIpMissingWebServiceError(NotErrorFailureSourceTypeMixin, WebServiceError):
    code = 'consumer_ip.empty'


class YandexTokenInvalidWebServiceError(NotErrorFailureSourceTypeMixin, WebServiceError):
    code = 'yandex_token.invalid'


class YandexSessionInvalidWebServiceError(NotErrorFailureSourceTypeMixin, WebServiceError):
    code = 'yandex_session.invalid'


class ApplicationUnknownWebServiceError(NotErrorFailureSourceTypeMixin, WebServiceError):
    code = 'application.unknown'


class ProviderTokenInvalidWebServiceError(NotErrorFailureSourceTypeMixin, WebServiceError):
    code = 'provider_token.invalid'


class ProviderTokenNotFoundWebServiceError(NotErrorFailureSourceTypeMixin, WebServiceError):
    code = 'provider_token.not_found'


class ProfileNotAllowedWebServiceError(NotErrorFailureSourceTypeMixin, WebServiceError):
    code = 'profile.not_allowed'


class ProfileBlockingExistsWebServiceError(NotErrorFailureSourceTypeMixin, WebServiceError):
    code = 'profile.blocking_exists'


class AccountNotFoundWebServiceError(NotErrorFailureSourceTypeMixin, WebServiceError):
    code = 'account.not_found'


class AccountInvalidTypeWebServiceError(NotErrorFailureSourceTypeMixin, WebServiceError):
    code = 'account.invalid_type'


class RateLimitExceededWebServiceError(NotErrorFailureSourceTypeMixin, WebServiceError):
    code = 'rate_limit.exceeded'


class _BaseTvmUserTicketInvalidWebServiceError(NotErrorFailureSourceTypeMixin, WebServiceError):
    code = 'user_ticket.invalid'


class TvmUserTicketInvalidWebServiceError(_BaseTvmUserTicketInvalidWebServiceError):
    description = 'Invalid user ticket'


class TvmUserTicketMissingScopesWebServiceError(_BaseTvmUserTicketInvalidWebServiceError):
    description = 'User ticket does not have required scopes'

    def __init__(self, ticket_scopes, missing_scopes):
        super(TvmUserTicketMissingScopesWebServiceError, self).__init__()
        self.ticket_scopes = ticket_scopes
        self.missing_scopes = missing_scopes


class TvmUserTicketNoUidWebServiceError(_BaseTvmUserTicketInvalidWebServiceError):
    description = 'User ticket does not contain uid'

    def __init__(self, known_uids):
        super(TvmUserTicketNoUidWebServiceError, self).__init__()
        self.known_uids = known_uids


class _InternalWebServiceError(WebServiceError):
    """
    Этот отказ объединяет в себя все временные отказы подсистем (которые
    лечатся повторной попыткой или может быть лечатся).
    Важно уточнить причину отказа в description, чтобы разработчики понимали
    куда идти с проблемой.
    """
    code = 'internal_error'


class DatabaseFailedWebServiceError(DatabaseFailureSourceTypeMixin, _InternalWebServiceError):
    description = 'Database failed'


class ProviderFailedWebServiceError(ExternalFailureSourceTypeMixin, _InternalWebServiceError):
    description = 'Provider failed'


class BlackboxFailedWebServiceError(NetworkFailureSourceTypeMixin, _InternalWebServiceError):
    description = 'Blackbox failed'


class KolmogorFailedWebServiceError(NetworkFailureSourceTypeMixin, _InternalWebServiceError):
    description = 'Kolmogor failed'


class PassportFailedWebServiceError(NetworkFailureSourceTypeMixin, _InternalWebServiceError):
    description = 'Passport failed'


class BillingApiFailedWebServiceError(NetworkFailureSourceTypeMixin, _InternalWebServiceError):
    description = 'Billing failed'


class NetworkFailedWebServiceError(NetworkFailureSourceTypeMixin, _InternalWebServiceError):
    description = 'Network failed'
