# -*- coding: utf-8 -*-

import datetime
from functools import partial
import logging
from logging.config import dictConfig as logging_dict_config
from random import SystemRandom

from flask import (
    Flask,
    request,
)
from passport.backend.core.lazy_loader import LazyLoader
from passport.backend.core.tvm.tvm_credentials_manager import TvmCredentialsManager
from passport.backend.social.common import (
    middleware,
    redis_client,
)
from passport.backend.social.common.builders.kolmogor import Kolmogor
from passport.backend.social.common.chrono import now
from passport.backend.social.common.db.utils import (
    build_master_db_config,
    build_slave_db_config,
    create_engine,
)
from passport.backend.social.common.exception import FailureSourceType
from passport.backend.social.common.grants import GrantsConfig
from passport.backend.social.common.limits import QLimits
from passport.backend.social.common.misc import dump_to_json_string
from passport.backend.social.common.provider_settings import providers
from passport.backend.social.common.social_config import social_config
from passport.backend.social.common.tvm import SocialTvmCredentialsManager
from passport.backend.social.common.useragent import build_http_pool_manager
from passport.backend.social.common.web_service import (
    Request,
    Response,
)
from passport.backend.social.proxy2 import logging_settings
from passport.backend.social.proxy2.error_handler import ErrorHandler
from passport.backend.social.proxy2.exception import InternalError
from passport.backend.social.proxy2.ping import ping_view
from passport.backend.social.proxy2.views.v1 import views
import passport.backend.social.proxy2.views.v2.redirect_to_profile
import passport.backend.social.proxylib
import werkzeug.exceptions


logger = logging.getLogger(__name__)


def prepare_interprocess_environment():
    """
    Инициализация подсистем, которые могут быть общими для нескольких процессов.

    Здесь следует инициализировать только подсистемы, которые не меняют своего
    состояния на протяжении жизни приложения.
    """
    logging.captureWarnings(True)
    logging_dict_config(logging_settings.LOGGING)

    social_config.init()

    LazyLoader.register('qlimits', QLimits)
    LazyLoader.get_instance('qlimits')

    providers.init()
    passport.backend.social.proxylib.init()


def prepare_intraprocess_environment():
    """
    Инициализация подсистем, которые нельзя делать общими для нескольких
    процессов.

    Здесь следует инициализировать подсистемы, которые меняют своё состояние
    на протяжении жизни приложения.
    """
    LazyLoader.register('chrono', lambda: datetime.datetime)
    LazyLoader.get_instance('chrono')

    LazyLoader.register('randomizer', SystemRandom)
    LazyLoader.get_instance('randomizer')

    LazyLoader.register('http_pool_manager', build_http_pool_manager)
    http_pool_manager = LazyLoader.get_instance('http_pool_manager')

    read_conn = create_engine(build_slave_db_config(social_config), False)
    LazyLoader.register('slave_db_engine', lambda: read_conn)
    LazyLoader.get_instance('slave_db_engine')

    write_conn = create_engine(build_master_db_config(social_config), False)
    LazyLoader.register('master_db_engine', lambda: write_conn)
    LazyLoader.get_instance('master_db_engine')

    redis_client.RedisClient.init()
    LazyLoader.register('redis', redis_client.RedisClient)
    LazyLoader.get_instance('redis')

    tvm_credentials_manager = SocialTvmCredentialsManager()
    LazyLoader.register('TvmCredentialsManager', lambda: tvm_credentials_manager)
    LazyLoader.get_instance('TvmCredentialsManager')

    grants_config = GrantsConfig(
        'social-proxy',
        backup_tvm_credentials_manager=TvmCredentialsManager(
            cache_time=social_config.proxy2_tvm_cache_time,
            keyring_config_name=social_config.proxy2_tvm_keyring_config_name,
            ticket_parser2_blackbox_env=social_config.tvm_environment,
        ),
        tvm_credentials_manager=tvm_credentials_manager,
    )
    LazyLoader.register('grants_config', lambda: grants_config)
    LazyLoader.get_instance('grants_config')

    kolmogor = Kolmogor(http_pool_manager)
    LazyLoader.register('kolmogor', lambda: kolmogor)
    LazyLoader.get_instance('kolmogor')


def create_app():
    app = Flask(__name__)
    app.request_class = Request
    app.response_class = Response

    def add_url_rule(path, methods, view_func, by_profile=True, by_task=True, by_token=True,
                     by_application=False):
        if by_profile:
            by_profile_path = '/proxy2/profile/<int:profile_id>' + path
            app.add_url_rule(by_profile_path, methods=methods, view_func=view_func)
        if by_task:
            by_task_path = '/proxy2/task/<task_id>' + path
            app.add_url_rule(by_task_path, methods=methods, view_func=view_func)
        if by_token:
            by_token_path = '/proxy2/token' + path
            app.add_url_rule(by_token_path, methods=methods, view_func=view_func)
        if by_application:
            by_app_path = '/proxy2/application/<app_name>' + path
            app.add_url_rule(by_app_path, methods=methods, view_func=view_func)

    add_url_rule(
        '', methods=['GET'],
        view_func=views.profile_view.as_view())

    add_url_rule(
        '/friends', methods=['GET'],
        view_func=views.friends_view.as_view())
    add_url_rule(
        '/friends/requests/count', methods=['GET'],
        view_func=views.friend_requests_count_view.as_view())

    add_url_rule(
        '/groups', methods=['GET'],
        view_func=views.groups_view.as_view())

    add_url_rule(
        '/counters', methods=['GET'],
        view_func=views.counters_view.as_view())
    add_url_rule(
        '/messages/unread/count', methods=['GET'],
        view_func=views.messages_unread_count_view.as_view())
    add_url_rule(
        '/wall/post', methods=['POST'],
        view_func=views.wall_post_view.as_view())
    add_url_rule(
        '/notifications/unread/count', methods=['GET'],
        view_func=views.notifications_unread_count_view.as_view())
    add_url_rule(
        '/mails/unread/count', methods=['GET'],
        view_func=views.mails_unread_count_view.as_view())
    add_url_rule(
        '/marks/unread/count', methods=['GET'],
        view_func=views.marks_unread_count_view.as_view())

    add_url_rule(
        '/photos', methods=['GET'],
        view_func=views.photos_view.as_view())
    add_url_rule(
        '/user_photos', methods=['GET'],
        view_func=views.user_photos_view.as_view())
    add_url_rule(
        '/photo_albums', methods=['GET'],
        view_func=views.photo_albums_view.as_view())
    add_url_rule(
        '/photo_albums/create', methods=['POST'],
        view_func=views.photo_album_create_view.as_view())
    add_url_rule(
        '/photo_post/get_request', methods=['GET'],
        view_func=views.photo_post_get_request_view.as_view())
    add_url_rule(
        '/photo_post/commit', methods=['POST'],
        view_func=views.photo_post_commit_view.as_view())

    # TODO Нужно понять использует ли кто-то эту ручку
    add_url_rule(
        '/sign_request', methods=['GET', 'POST'],
        view_func=views.sign_request.as_view())

    add_url_rule(
        '/refresh_token', methods=['POST'],
        view_func=views.refresh_token_view.as_view(),
        by_profile=False,
        by_task=False,
        by_token=False,
        by_application=True)

    app.add_url_rule(
        '/proxy2/ping', methods=['GET'],
        view_func=ping_view)

    app.add_url_rule(
        '/proxy2/redirect_to_profile', methods=['GET'],
        view_func=passport.backend.social.proxy2.views.v2.redirect_to_profile.RedirectToProfile.as_view(),
    )

    app.before_request(middleware.setup_request)
    app.before_request(middleware.log_http_request)
    app.before_request(middleware.log_api_call)
    app.after_request(partial(middleware.after_request, 'social-proxy2-access-log'))
    app.errorhandler(Exception)(handle_exception)
    app.errorhandler(404)(handle_page_not_found)
    app.errorhandler(werkzeug.exceptions.MethodNotAllowed)(handle_page_not_found)

    return app


def handle_page_not_found(_):
    response = {
        'task': {
            'state': 'failure',
            'reason': {
                'code': 'page_not_found',
                'type': FailureSourceType.external,
            },
        },
    }
    json_response = dump_to_json_string(response)
    return Response(json_response, status=404, mimetype='application/json')


def handle_exception(_):
    logger.error('Unhandled exception', exc_info=True)
    internal_error = InternalError('Unhandled exception')
    error_handler = ErrorHandler(internal_error)
    error_handler.exception_to_graphite()
    response = {'task': error_handler.exception_to_response()}
    if hasattr(request, 'started_at'):
        duration = now.f() - request.started_at
        response['task']['runtime'] = duration
    json_response = dump_to_json_string(response)
    http_response = Response(json_response, status=500, mimetype='application/json')

    http_response.api_status = 'error'
    reason = response['task'].get('reason')
    http_response.api_error_code = reason.get('code') if reason else None

    return http_response
