# -*- coding: utf-8 -*-

import logging

from passport.backend.social.common.context import request_ctx
from passport.backend.social.common.exception import (
    AlbumNotExistsProxylibError,
    DatabaseError,
    FailureSourceType,
    InvalidTokenProxylibError,
    PermissionProxylibError,
    ProviderCommunicationProxylibError,
    ProviderRateLimitExceededProxylibError,
    SocialUserDisabledProxylibError,
)
from passport.backend.social.proxy2.exception import (
    InternalError,
    ProxyError,
    RateLimitExceededError,
)
from passport.backend.social.proxylib.error_handler import ErrorHandler as _ErrorHandler


logger = logging.getLogger(__name__)


class ErrorHandler(_ErrorHandler):
    def exception_to_response(self):
        ex = self._exception
        output = {'state': 'failure'}
        if isinstance(ex, ProviderCommunicationProxylibError):
            output['reason'] = {
                'description': unicode(ex.description or ex.message or ''),
                'code': 'api_error',
                'message': unicode(ex.message or ''),
            }
        elif isinstance(ex, ProxyError):
            output['reason'] = {
                'description': ex.description,
                'code': ex.code,
            }
        elif isinstance(ex, InvalidTokenProxylibError):
            output['reason'] = {
                'description': 'User cannot be authenticated using existing tokens',
                'code': 'invalid_token',
            }
        elif isinstance(ex, PermissionProxylibError):
            output['reason'] = {
                'description': 'There is no token with sufficient permissions to complete the request',
                'code': 'permission_error',
            }
        elif isinstance(ex, ProviderRateLimitExceededProxylibError):
            output['reason'] = {
                'description': 'Per-user or per-application rate limit exceeded',
                'code': RateLimitExceededError.code,
            }
        elif isinstance(ex, SocialUserDisabledProxylibError):
            output['reason'] = {
                'description': 'User is disabled in social provider',
                'code': 'user_disabled',
            }
        elif isinstance(ex, AlbumNotExistsProxylibError):
            output['reason'] = {
                'description': 'Requested album does not exist.',
                'code': 'album_not_exists',
            }
        elif isinstance(ex, DatabaseError):
            output['reason'] = {
                'code': InternalError.code,
                'description': 'Database failed',
            }
        else:
            return

        output['reason'].setdefault('message', unicode(ex or ''))

        error_type = self._exception_to_error_type()
        if error_type not in {FailureSourceType.external, FailureSourceType.internal}:
            if error_type in {FailureSourceType.not_error, FailureSourceType.network}:
                error_type = FailureSourceType.external
            else:
                error_type = FailureSourceType.internal
        output['reason']['type'] = error_type

        provider = getattr(request_ctx, 'provider', None)
        if provider:
            output['provider'] = provider['name']

        return output
