# -*- coding: utf-8 -*-

from collections import (
    defaultdict,
    namedtuple,
)
import logging
from operator import attrgetter

from passport.backend.social.common.db.execute import execute
from passport.backend.social.common.db.schemas import (
    business_application_map_table as bamt,
    profile_table as pt,
)
from passport.backend.social.common.db.utils import get_slave_engine
from passport.backend.social.common.misc import (
    FACEBOOK_BUSINESS_ID,
    get_business_userid,
)
from passport.backend.social.common.refresh_token.utils import filter_refreshable_tokens
from passport.backend.social.common.serialize import serialize_scopes
from passport.backend.social.common.social_config import social_config
from passport.backend.social.common.token.utils import (
    filter_tokens_granted_to_use,
    find_all_tokens_for_profile,
)
from passport.backend.social.proxy2.utils import filter_allowed_grants
from passport.backend.utils.common import chop
from sqlalchemy import sql
from sqlalchemy.sql.expression import select


logger = logging.getLogger('social.proxy2')

# При работе через task, у нас нет настоящего профиля, полученного из базы, но мы
# можем сконструировать и использовать похожий на него объект.
FakeProfile = namedtuple('FakeProfile', 'provider_id userid username uid profile_id')


def get_profile(profile_id):
    query = select([pt], from_obj=[pt]).where(pt.c.profile_id == profile_id)
    profile = execute(get_slave_engine(), query).fetchone()
    return profile


def get_tokens(profile_id,  application_ids=None):
    """
    Возвращает список токенов, такой что для каждого уникального
    разрешения (scope), выбирается самый свежий токен.

    Входные параметры:
        profile_id
            Идентификатор профиля
        application_ids
            Выбрать только те токены, которые были выданы данным приложениям.
    """
    logger.info('Getting tokens...')

    access_tokens = find_all_tokens_for_profile(profile_id, get_slave_engine(), application_ids)
    access_tokens = filter_tokens_granted_to_use(
        access_tokens,
        filter_allowed_grants_func=filter_allowed_grants,
    )

    logger.debug('Total tokens fetched: %d' % len(access_tokens))

    not_expired_access_tokens = []
    expired_tokens = []
    for access_token in access_tokens:
        if access_token.is_going_to_expire():
            expired_tokens.append(access_token)
        else:
            not_expired_access_tokens.append(access_token)
    logger.debug('Not expired tokens fetched: %d' % len(not_expired_access_tokens))

    expired_refreshable_access_tokens = filter_refreshable_tokens(expired_tokens, get_slave_engine())
    logger.debug('Expired but refreshable tokens fetched: %d' % len(expired_refreshable_access_tokens))

    # выбираем токены с уникальными scope наилучшей свежести
    tokens = []
    scopes_for_tokens = set()
    for token_list in [not_expired_access_tokens, expired_refreshable_access_tokens]:
        for token in sorted(token_list, key=attrgetter('created'), reverse=True):
            scopes = tuple(sorted(token.scopes))
            if scopes not in scopes_for_tokens:
                tokens.append(token)
                scopes_for_tokens.add(scopes)
    logger.debug('Tokens scopes available: %s' % ','.join([serialize_scopes(t.scopes) for t in tokens]))

    logger.debug('Resulting number of tokens: %d' % len(tokens))
    return tokens


def get_profiles_by_provider_userids(provider_id, userids):
    logger.info('Getting profiles for provider by userids')

    profile_rows = list()
    for userids_chunk in chop(userids, social_config.max_sql_in_function_values):
        query = (
            select([pt])
            .where(
                sql.and_(
                    pt.c.provider_id == provider_id,
                    pt.c.userid.in_(userids_chunk)
                ),
            )
        )
        rows = execute(get_slave_engine(), query).fetchall()
        profile_rows.extend(rows)

    logger.debug('Profiles fetched count: %d' % len(profile_rows))

    profiles = defaultdict(list)
    for profile_row in profile_rows:
        profile = dict()
        for key in ['uid', 'profile_id', 'userid', 'username']:
            profile[key] = getattr(profile_row, key)
        profiles[profile_row.userid].append(profile)

    return profiles


def simple_userids_2_business_userids(simple_userids):
    """
    Строит отображение user_id -> business_userid.

    Выходные параметры
        Двойка (simple_userid_2_business_userid, unknown_simple_userids)

        unknown_simple_userids
            Список userids для которых не нашлось business_id, но это не
            означает, что для них нет социального профиля. Потому что в момент
            преобразования userid в business_id у нас могло не быть возможности
            узнать bussiness_id пользователя, из-за нехватки токена.

    Примечания
        Для простоты получаем business_userid только для FACEBOOK_BUSINESS_ID.
    """
    records = list()
    for userids_chunk in chop(simple_userids, social_config.max_sql_in_function_values):
        query = select([bamt]).where(
            (bamt.c.userid.in_(userids_chunk)) &
            (bamt.c.business_id == FACEBOOK_BUSINESS_ID),
        )
        records.extend(execute(get_slave_engine(), query).fetchall())

    simple_userid_2_business_userid = {}
    for rec in records:
        business_userid = get_business_userid(FACEBOOK_BUSINESS_ID, rec.business_token)
        simple_userid_2_business_userid[rec.userid] = business_userid

    unknown_simple_userids = set(simple_userids) - set(r.userid for r in records)

    return simple_userid_2_business_userid, list(unknown_simple_userids)
