# -*- coding: utf-8 -*-

import logging

from passport.backend.social.common.context import request_ctx
from passport.backend.social.common.exception import (
    GrantsMissingError as _GrantsMissingError,
    RateLimitExceededError as _RateLimitExceededError,
    TvmTicketParsingGrantsContextError,
)
from passport.backend.social.common.grants import (
    check_any_of_grants,
    filter_allowed_grants as _filter_allowed_grants,
    find_all_consumers_by_context,
    get_grants_config,
    GrantsContext,
)
from passport.backend.social.common.throttler import Throttler
from passport.backend.social.common.web_service import description_for_consumer_from_grants_missing_error
from passport.backend.social.proxy2.exception import (
    GrantsMissingError,
    InvalidParametersError,
    RateLimitExceededError,
)


logger = logging.getLogger(__name__)


def get_grants_context(request):
    return GrantsContext(
        consumer_ip=request.consumer_ip,
        consumer=request.values.get('consumer'),
        ticket_body=request.ticket_body,
    )


def filter_allowed_grants(grants):
    grants_config = get_grants_config()
    grants_config.load()
    return _filter_allowed_grants(grants_config, request_ctx.grants_context, grants)


def get_request_args(context, from_form=False, exclude=None):
    if from_form:
        source = context['request'].form
    else:
        source = context['request'].args

    return dict(
        (key, value)
        for key, value in source.iteritems()
        if key not in (exclude or [])
    )


def get_request_arg(context, arg_name, is_required=False, from_form=False):
    if from_form:
        source = context['request'].form
    else:
        source = context['request'].args

    value = source.get(arg_name) or context['kwargs'].get(arg_name)

    if not value:
        if is_required:
            raise InvalidParametersError('`%s` is a required parameter' % arg_name)
        return
    return value


def get_bool_request_arg(context, arg_name, is_required=False, from_form=False, default_value=None):
    value = get_request_arg(context, arg_name, is_required=is_required, from_form=from_form)
    if value is not None:
        return value.lower() in ['1', 'yes', 'y', 'true']
    else:
        return default_value


def throttle_proxy(proxy_method_name, request):
    grants_config = get_grants_config()
    grants_config.load()
    context = get_grants_context(request)
    request_name = 'proxy:' + proxy_method_name
    try:
        throttler = Throttler.from_grants_context(request_name, grants_config, context)
        throttler.throttle()
    except _RateLimitExceededError as e:
        raise RateLimitExceededError(str(e))


class Grant(object):
    def __init__(self, grant_name, condition_func=None):
        self.grant_name = grant_name
        self.condition_func = condition_func

    def check(self, request, grants_context):
        if self.condition_func:
            if not self.condition_func(request):
                return

        Grant.check_any([self.grant_name], grants_context)

    @staticmethod
    def check_any(grants, grants_context):
        grants_config = get_grants_config()
        grants_config.load()

        try:
            allowed_grants = check_any_of_grants(grants_config, grants_context, grants)
        except _GrantsMissingError as e:
            logger.warning('Access denied (%s)' % str(e))
            raise GrantsMissingError(description_for_consumer_from_grants_missing_error(e))

        logger.info('Access to %s is granted for %s' % (allowed_grants, str(grants_context)))


def format_response(mapping, response):
    output = {}
    for k, v in mapping.iteritems():
        if isinstance(v, basestring):
            if v in response:
                output[k] = response[v]
        else:  # dict
            _out = format_response(v, response)
            if _out:
                output[k] = _out
    return output


def guess_consumer_name(request):
    """
    Угадывает имя потребителя сделавшего запрос. К сожалению имя
    потребителя мы можем только угадать, т.к. social proxy есть
    возможность не передавать его явно, в этом случае потребитель
    подбирается по IP.
    """
    consumer = None
    grants_config = get_grants_config()
    grants_config.load()
    grants_context = get_grants_context(request)
    try:
        consumers = find_all_consumers_by_context(grants_config, grants_context)
    except TvmTicketParsingGrantsContextError:
        consumers = list()
    if consumers:
        consumer = consumers[0]
    return consumer
