# -*- coding: utf-8 -*-

from collections import namedtuple
from functools import partial
import logging

from flask import request
from passport.backend.social.common.chrono import now
from passport.backend.social.common.context import request_ctx
from passport.backend.social.common.exception import FailureSourceType
from passport.backend.social.common.misc import (
    dump_to_json_string,
    parse_userid,
    USERID_TYPE_SIMPLE,
)
from passport.backend.social.common.provider_settings import providers
from passport.backend.social.common.redis_client import get_redis
from passport.backend.social.common.task import load_task_from_redis
from passport.backend.social.common.token.domain import Token
from passport.backend.social.common.token.utils import get_grants_to_use_token
from passport.backend.social.common.web_service import Response
from passport.backend.social.proxy2.error_handler import ErrorHandler
from passport.backend.social.proxy2.exception import (
    InternalError,
    InvalidParametersError,
    InvalidTaskError,
    ProfileNotFoundError,
    ProxyMethodNotImplementedError,
    TaskNotFoundError,
    TokensNotFoundError,
)
from passport.backend.social.proxy2.misc import (
    FakeProfile,
    get_profile,
    get_tokens,
)
from passport.backend.social.proxy2.utils import (
    get_grants_context,
    Grant,
    throttle_proxy,
)
from passport.backend.social.proxylib import get_proxy


logger = logging.getLogger(__name__)


class MultiResult(object):
    """
    Если вьюшка вернула ответ в таком формате, то result будет положен в ответе в поле "result",
    а все данные из словаря additional_fields будут также положены рядом.
    """
    def __init__(self, result, additional_fields=None):
        self.result = result
        self.additional_fields = additional_fields


ProxyResponse = namedtuple('ProxyResponse', 'result raw_response')


class RequestProcessor(object):
    def __init__(self, func, proxy_method_name, grants, request, kwargs):
        self._grants_context = request_ctx.grants_context = get_grants_context(request)
        self.check_grants(grants, request, self._grants_context)
        throttle_proxy(proxy_method_name, request)

        self.func = func
        self.proxy_method_name = proxy_method_name

        self._request = request
        self._kwargs = kwargs
        self._data_source = None
        self._other_process_data = None
        self._provider_id = None
        self._provider = None
        self._app = None

        if self._grants_context.matching_consumers:
            self._consumer = self._grants_context.matching_consumers[0]
        else:
            self._consumer = request.values.get('consumer')

    def check_grants(self, grants, request, grants_context):
        logger.debug('Checking grants...')
        for grant in grants:
            grant.check(request, grants_context)

    @property
    def _tokens(self):
        raise NotImplementedError()  # pragma: no cover

    @property
    def _request_identifier(self):
        raise NotImplementedError()  # pragma: no cover

    @property
    def _profile(self):
        raise NotImplementedError()  # pragma: no cover

    @property
    def _simple_userid(self):
        raise NotImplementedError()  # pragma: no cover

    def _setup_provider(self):
        if self._provider_id is None:
            return

        provider = providers.get_provider_info_by_id(self._provider_id)
        if not provider:
            logger.error('Can not find provider info for provider_id=%s' % self._provider_id)
            raise InternalError('Unknown provider')

        request_ctx.provider = provider
        return provider

    def _assert_method_availability(self):
        if self._provider:
            logger.debug('Checking method availability for provider "%s"' % self._provider['code'])
            proxy = get_proxy(code=self._provider['code'])
        else:
            proxy = get_proxy(app=self._app)
        if self.proxy_method_name is not None and not hasattr(proxy, self.proxy_method_name):
            raise ProxyMethodNotImplementedError()

    def _log_called_method(self):
        logger.info('"%s" view with id %s' % (
            self.proxy_method_name or '<no_method>',
            self._request_identifier,
        ))

    def _compose_response(self, view_response):
        if isinstance(view_response, ProxyResponse):
            view_response = MultiResult(
                view_response.result,
                additional_fields={
                    'raw_response': view_response.raw_response,
                },
            )

        multi_mode = isinstance(view_response, MultiResult)
        output = {
            'result': view_response.result if multi_mode else view_response,
            'task': {
                'provider': None if not self._provider else self._provider['name'],
                'state': 'success',
            }
        }

        if multi_mode and view_response.additional_fields:
            update_args = dict((k, v) for k, v in view_response.additional_fields.iteritems() if v is not None)
            output.update(update_args)
        return output

    def _set_additional_response_fields(self, response_dict):
        pass

    def __call__(self):
        self._provider = self._setup_provider()
        self._assert_method_availability()
        self._log_called_method()

        context = dict(
            profile=self._profile,
            simple_userid=self._simple_userid,
            provider=self._provider,
            tokens=self._tokens,
            method=self.proxy_method_name,
            request=self._request,
            kwargs=self._kwargs,
            consumer=self._consumer,
        )

        # Непосредственное выполнение запроса.
        result = self.func(context)

        if isinstance(result, Response):
            return result

        output = self._compose_response(result)

        self._set_additional_response_fields(output)

        return output


class ProfileRequestProcessor(RequestProcessor):
    def __init__(self, func, proxy_method_name, grants, request, kwargs):
        super(ProfileRequestProcessor, self).__init__(func, proxy_method_name, grants, request, kwargs)
        profile_id = self._kwargs['profile_id']

        self._db_profile = get_profile(profile_id)
        if self._profile is None:
            raise ProfileNotFoundError('Profile with id="%s" not found!' % profile_id)

        self._provider_id = self._profile.provider_id

    def _get_allowed_application_ids(self):
        """
        Вернем список идентификаторов разрешенных приложений, если пользователь
        передал параметр allowed_applications, иначе None.
        Приложения можно задавать по имени.
        """
        app_names = filter(bool, self._request.args.get('allowed_applications', '').split(','))
        apps, unknown_apps = providers.get_many_applications_by_names(app_names)
        if unknown_apps:
            formatted_apps = ', '.join(map(str, unknown_apps))
            raise InvalidParametersError('Applications %s not found' % formatted_apps)
        return [a.identifier for a in apps]

    @property
    def _tokens(self):
        tokens = get_tokens(
            self._profile.profile_id,
            application_ids=self._get_allowed_application_ids(),
        )
        if not tokens:
            raise TokensNotFoundError()
        return tokens

    @property
    def _profile(self):
        return self._db_profile

    @property
    def _request_identifier(self):
        return self._profile.profile_id

    @property
    def _simple_userid(self):
        userid_type, _ = parse_userid(self._profile.userid)
        if userid_type == USERID_TYPE_SIMPLE:
            return self._profile.userid

    def _set_additional_response_fields(self, response_dict):
        response_dict['task']['profile_id'] = self._profile.profile_id


class TaskRequestProcessor(RequestProcessor):
    def __init__(self, func, proxy_method_name, grants, request, kwargs):
        super(TaskRequestProcessor, self).__init__(func, proxy_method_name, grants, request, kwargs)
        self._task_id = self._kwargs['task_id']
        self._task = self._get_task_from_redis(self._task_id, self._consumer)

        self._profile_dict = self._task.get_social_userinfo()
        self._provider_id = self._task.provider['id']

    @property
    def _tokens(self):
        return [self._task.get_token()]

    @property
    def _profile(self):
        return FakeProfile(
            provider_id=self._provider_id,
            userid=self._profile_dict['userid'],
            username=self._profile_dict.get('username'),
            uid=None,
            profile_id=None,
        )

    @property
    def _request_identifier(self):
        return self._task_id

    @property
    def _simple_userid(self):
        return self._profile_dict['userid']

    @staticmethod
    def _get_task_from_redis(task_id, consumer):
        """
        Получим task из redis, проверив все условия.
        """
        if not consumer:
            raise InvalidParametersError('`consumer` is a required parameter to do anything by task_id')

        task = load_task_from_redis(get_redis(), task_id)
        if not task:
            raise TaskNotFoundError('Task with id="%s" not found!' % task_id)

        Grant.check_any(get_grants_to_use_token(application=task.application), request_ctx.grants_context)

        if not task.consumer:
            raise InvalidTaskError('No `consumer` in task. Maybe broker has been used '
                                   'without `consumer` passed?')

        if task.consumer != consumer:
            raise InvalidTaskError('Current consumer (%s) does not match task consumer (%s)' %
                                   (consumer, task.consumer))
        return task


class AccessTokenRequestProcessor(RequestProcessor):
    def _get_application(self):
        application_name = self._request.args.get('application')

        if not application_name:
            raise InvalidParametersError('`application` GET parameter not found')

        app = providers.get_application_by_name(application_name)
        if not app:
            raise InvalidParametersError('Application `%s` not found' % application_name)

        Grant.check_any(get_grants_to_use_token(application=app), request_ctx.grants_context)
        return app

    def __init__(self, func, proxy_method_name, grants, request, kwargs):
        super(AccessTokenRequestProcessor, self).__init__(func, proxy_method_name, grants, request, kwargs)
        self._access_token = self._request.header_provider_token
        self._access_token_secret = self._request.header_provider_token_secret

        self._app = self._get_application()
        self._provider_id = self._app.provider['id'] if self._app.provider else None

    @property
    def _tokens(self):
        return [
            Token(
                application=self._app,
                application_id=self._app.identifier,
                value=self._access_token,
                secret=self._access_token_secret,
            ),
        ]

    @property
    def _profile(self):
        return

    @property
    def _request_identifier(self):
        return '<token>'

    @property
    def _simple_userid(self):
        # Не знаем userid, будем жить без него.
        return


class ApplicationRequestProcessor(RequestProcessor):
    """
    Процессор для методов, которым нужен токен приложения, а не пользователя
    """
    def _get_application(self):
        app_name = self._kwargs['app_name']

        app = providers.get_application_by_name(app_name)
        if not app:
            raise InvalidParametersError('Application `%s` not found' % app_name)

        Grant.check_any(get_grants_to_use_token(application=app), request_ctx.grants_context)
        return app

    def __init__(self, func, proxy_method_name, grants, request, kwargs):
        super(ApplicationRequestProcessor, self).__init__(func, proxy_method_name, grants, request, kwargs)

        self._app = self._get_application()
        self._provider_id = None if not self._app.provider else self._app.provider['id']

    @property
    def _tokens(self):
        # Возвращаем фейковый токен, для совместимости с текущим кодом прокси.
        # В ручки провайдера он передаваться не будет.
        return [
            Token(
                application=self._app,
                application_id=self._app.identifier,
            ),
        ]

    @property
    def _profile(self):
        return

    @property
    def _request_identifier(self):
        return '<application>'

    @property
    def _simple_userid(self):
        # Юзера тут нет совсем
        return


class choose_request_processor(object):
    """
    По сути, фабрика RequestProcessor'ов.
    """
    def __init__(self, func, proxy_method_name, grants):
        self.func = func
        self.original_method = func
        self.grants = grants
        self.proxy_method_name = proxy_method_name
        self.__name__ = func.__name__

    @staticmethod
    def _get_processor_class(request, **kwargs):
        logger.debug('Detecting operation mode...')

        if kwargs.get('profile_id'):
            processor_class = ProfileRequestProcessor
        elif kwargs.get('task_id'):
            processor_class = TaskRequestProcessor
        elif kwargs.get('app_name'):
            processor_class = ApplicationRequestProcessor
        elif request.header_provider_token:
            processor_class = AccessTokenRequestProcessor
        else:
            raise InvalidParametersError('Bad request. No `profile_id`, `task_id`, access token or application passed.')

        return processor_class

    def __call__(self, request, **kwargs):
        """
        proxy умеет работать по profile_id, брокерному task, по непосредственно переданному в хедерах токену
        или по приложению (для которого будет получен токен приложения).
        Определим режим работы и вернем нужные данные.
        Возвращаем (<режим>, provider_id, <прочие данные>)
        """
        processor_class = self._get_processor_class(request, **kwargs)

        processor = processor_class(self.func, self.proxy_method_name, self.grants, request, kwargs)
        return processor()

    def as_view(self):
        view_func = partial(self._view_func, self)
        view_func.__name__ = self.func.__name__
        return view_func

    @staticmethod
    def _view_func(self, *args, **kwargs):
        status_code = 200
        api_status = None
        api_error_code = None

        try:
            response = self(request, *args, **kwargs)
        except Exception as e:
            logger.info('Processing exception %s' % type(e).__name__)
            error_handler = ErrorHandler(e)
            response = error_handler.exception_to_response()
            if not response:
                raise
            response = {'task': response}

            if 'reason' not in response['task']:
                status_code = 500
            elif response['task']['reason'].get('type') == FailureSourceType.internal:
                status_code = 500

            api_status = 'error'
            reason = response['task'].get('reason')
            api_error_code = reason.get('code') if reason else None

            error_handler.exception_to_graphite()

        response['task']['runtime'] = now.f() - request.started_at

        response = dump_to_json_string(response)
        http_response = Response(response, status=status_code, mimetype='application/json')
        http_response.api_status = api_status
        http_response.api_error_code = api_error_code
        return http_response
