# -*- coding: utf-8 -*-

import logging
import urllib

from passport.backend.social.common.exception import (
    AlbumNotExistsProxylibError,
    BasicProxylibError,
)
from passport.backend.social.common.provider_settings import get_profile_addresses
from passport.backend.social.common.providers.Facebook import Facebook
from passport.backend.social.common.web_service import Response
from passport.backend.social.proxy2.error_handler import ErrorHandler
from passport.backend.social.proxy2.exception import (
    InvalidParametersError,
    ProxyError,
)
from passport.backend.social.proxy2.misc import (
    get_profiles_by_provider_userids,
    simple_userids_2_business_userids,
)
from passport.backend.social.proxy2.statbox import StatboxFriendsLogger
from passport.backend.social.proxy2.utils import (
    format_response,
    get_bool_request_arg,
    get_request_arg,
    get_request_args,
    Grant,
)
from passport.backend.social.proxy2.views.v1.request_processors import (
    choose_request_processor,
    MultiResult,
    ProxyResponse,
)
from passport.backend.social.proxylib import get_proxy
from passport.backend.social.proxylib.method_caller import MultitokenProxyMethodCaller


logger = logging.getLogger('social.proxy2.views')


PERSON_FIELDS = {
    'name': {'first': 'firstname', 'last': 'lastname', 'middle': 'middlename'},
    'locale': 'locale',
    'gender': 'gender',
    'email': 'email',
    'avatar': 'avatar',
    'birthdate': 'birthday',
    'location': {
        'city': {
            'name': 'city_name',
            'geobase_id': 'city_id',
        },
        'country': {
            'geobase_id': 'country_id',
            'code': 'country_code'
        },
        'timezone': {
            'offset': 'timezone_offset',
            'name': 'timezone_name',
        },
    },
    'devices': 'devices',
    'education': 'education',
    'hometown': 'hometown',
    'relationship_status': 'relationship_status',
    'work': 'work',
    'followers_count': 'followers_count',
    'friends_count': 'friends_count',
    'messages_count': 'messages_count',
    'phone_number': 'phone',
}

ACCOUNT_MAPPING = {
    'updated': 'updated',
    'is_verified': 'is_verified',
    'userid': 'userid',
    'username': 'username',
    'person': PERSON_FIELDS,
}
GROUP_MAPPING = {
    'gid': 'gid',
    'name': 'name',
    'privacy': 'privacy',
    'avatar': 'avatar',
}


def view(proxy_method_name, grants):
    def decorator(func):
        return choose_request_processor(func, proxy_method_name, grants)
    return decorator


def get_proxy_response(context, refresh_tokens_enabled=True, **kwargs):
    proxy_method_caller = MultitokenProxyMethodCaller(context['tokens'], refresh_tokens_enabled=refresh_tokens_enabled)
    result = proxy_method_caller.call_proxy_method(context['method'], dict(kwargs, userid=context['simple_userid']))

    # Добавим оригинальный ответ соц сети
    raw_response_obj = proxy_method_caller.last_called_proxy.r.context.get('raw_response')
    raw_response = raw_response_obj.decoded_data if raw_response_obj else None

    return ProxyResponse(result, raw_response)


@view('get_profile', [Grant('profile'), Grant('friends', lambda request: request.form.get('friends')),
                      Grant('groups', lambda request: request.form.get('groups'))])
def profile_view(context):
    with_friends = get_bool_request_arg(context, 'friends')
    with_groups = get_bool_request_arg(context, 'groups')

    response, raw_response = get_proxy_response(context)
    account = format_response(ACCOUNT_MAPPING, response)
    profile = context.get('profile')
    if profile:
        uid = profile.uid
        profile_id = profile.profile_id
    else:
        uid = None
        profile_id = None
    addresses = get_profile_addresses(
        provider_id=context['provider']['id'],
        userid=account.get('userid'),
        username=account.get('username'),
        uid=uid,
        profile_id=profile_id,
    )
    output = {
        'account': account,
        'addresses': addresses,
    }

    if with_friends:
        new_context = context.copy()
        new_context['method'] = 'get_friends'
        output.update(friends_view.original_method(new_context))

    if with_groups:
        new_context = context.copy()
        new_context['method'] = 'get_groups'
        output.update(groups_view.original_method(new_context))

    return ProxyResponse(output, raw_response)


@view('get_counters', [Grant('counters_all')])
def counters_view(context):
    provider_code = context['provider']['code']
    proxy = get_proxy(provider_code)
    output = {}
    for attr in dir(proxy):
        if not attr.endswith('_count'):
            continue
        context['method'] = attr
        try:
            result = {'state': 'ok', 'result': get_proxy_response(context).result}
        except (ProxyError, BasicProxylibError) as ex:
            result = ErrorHandler(ex).exception_to_response()
            if result is None:
                raise
        if attr.startswith('get_'):
            attr = attr[4:]
        output[attr] = result
    return output


@view('get_friends', [Grant('friends')])
def friends_view(context):
    public_only = get_bool_request_arg(context, 'public_only')
    response, raw_response = get_proxy_response(context, public_only=public_only)

    friends = []
    for friend_data in response:
        friend = format_response(ACCOUNT_MAPPING, friend_data)
        friends.append(friend)

    with_yandex_profiles = get_bool_request_arg(context, 'yandex_profiles')
    if with_yandex_profiles:
        provider_id = context['provider']['id']
        userids = [x['userid'] for x in friends]

        if provider_id == Facebook.id:
            # Нужно преобразовать simple_userid в business_userid,
            # т.к. хранятся в БД именно business_userid.
            simple_userid_2_business_id, unknown_simple_userids = simple_userids_2_business_userids(userids)
            userids = simple_userid_2_business_id.values() + unknown_simple_userids

        accounts = get_profiles_by_provider_userids(provider_id, userids)

        for friend in friends:
            userid = friend['userid']
            if provider_id == Facebook.id:
                userid = simple_userid_2_business_id.get(userid, userid)

            friend['yandex_profiles'] = accounts.get(userid, [])

    _friends_to_statbox(context, friends, with_yandex_profiles)

    return ProxyResponse({'friends': friends}, raw_response)


def _friends_to_statbox(context, friends, with_yandex_profiles):
    profile = context.get('profile')
    if not profile:
        # Профиля нет, когда ручка вызывается по токену. В этом случае мы не
        # знаем профиля, а следовательно точного uid'а. Поэтому в лог ничего
        # не пишем.
        return

    master_uid = profile.uid
    if not master_uid:
        # Uid'а нет, когда ручка вызывает по task_id. В этом случае мы не
        # знаем профиля, а следовательно точного uid'а. Поэтому в лог ничего
        # не пишем.
        return

    master_userid = profile.userid
    consumer = context['consumer']
    provider = context['provider']

    if provider['id'] == Facebook.id:
        userids = [f['userid'] for f in friends]
        userids.append(master_userid)
        simple_userid_2_business_id = simple_userids_2_business_userids(userids)[0]
        master_userid = simple_userid_2_business_id.get(master_userid, master_userid)

    friends_userid_and_uids_list = list()
    for friend in friends:
        friend_profiles = friend.get('yandex_profiles', list())
        uids = [p['uid'] for p in friend_profiles]
        userid = friend['userid']
        friends_userid_and_uids_list.append((userid, uids))

    if provider['id'] == Facebook.id:
        for i, (userid, uids) in enumerate(friends_userid_and_uids_list):
            userid = simple_userid_2_business_id.get(userid, userid)
            friends_userid_and_uids_list[i] = (userid, uids)

    if with_yandex_profiles:
        userid_and_one_uid_list = list()
        for userid, uids in friends_userid_and_uids_list:
            uids = uids or ['']
            userid_and_one_uid_list.extend([(userid, u) for u in uids])

        friends_userids = [userid for userid, _ in userid_and_one_uid_list]
        friends_uids = [uid for _, uid in userid_and_one_uid_list]
    else:
        friends_userids = [userid for userid, _ in friends_userid_and_uids_list]
        friends_uids = []

    StatboxFriendsLogger().log(
        action='get_friends',
        uid=master_uid,
        provider=provider['name'],
        userid=master_userid,
        consumer=consumer,
        friends_userids=_friends_list_to_statbox_csv(friends_userids),
        friends_uids_enabled=bool(with_yandex_profiles),
        friends_uids=_friends_list_to_statbox_csv(friends_uids),
    )


def _friends_list_to_statbox_csv(items):
    s_items = list()
    for item in items:
        s_items.append(
            str(item)
            .replace('\\', '\\\\')
            .replace(',', '\\,')
        )
    return ','.join(s_items)


@view('get_groups', [Grant('groups'), Grant('members', lambda request: request.form.get('members'))])
def groups_view(context):
    response, raw_response = get_proxy_response(context)

    groups = []
    for group_data in response:
        group = format_response(GROUP_MAPPING, group_data)
        groups.append(group)

    return ProxyResponse({'groups': groups}, raw_response)


@view('get_friend_requests_count', [Grant('friend_requests_count')])
def friend_requests_count_view(context):
    return get_proxy_response(context)


@view('get_notifications_unread_count', [Grant('notifications_unread_count')])
def notifications_unread_count_view(context):
    return get_proxy_response(context)


@view('get_mails_unread_count', [Grant('mails_unread_count')])
def mails_unread_count_view(context):
    return get_proxy_response(context)


@view('get_marks_unread_count', [Grant('marks_unread_count')])
def marks_unread_count_view(context):
    return get_proxy_response(context)


@view('get_messages_unread_count', [Grant('messages_unread_count')])
def messages_unread_count_view(context):
    return get_proxy_response(context)


@view('wall_post', [Grant('wall_post')])
def wall_post_view(context):
    data = {}
    for key in ['text', 'picture', 'name', 'caption', 'description', 'link']:
        data[key] = get_request_arg(context, key, is_required=False, from_form=True)

    response = get_proxy_response(context, **data)
    return ProxyResponse(response.result, response.raw_response)


@view('get_stories', [Grant('stories')])
def get_stories_view(context):
    data = {
        'action_type': get_request_arg(context, 'action_type', is_required=True),
        'app_id_filter': get_request_arg(context, 'app_id_filter', is_required=False),
    }

    response = get_proxy_response(context, **data)
    return ProxyResponse(response.result, response.raw_response)


@view('publish_story', [Grant('stories')])
def publish_story_view(context):
    data = {}
    required_args = ['action_type', 'object_type', 'object_id']
    for key in required_args:
        data[key] = get_request_arg(context, key, is_required=True, from_form=True)

    data.update(get_request_args(context, from_form=True, exclude=required_args))

    response = get_proxy_response(context, **data)
    return ProxyResponse(response.result, response.raw_response)


@view('get_story', [Grant('stories')])
def get_story_view(context):
    data = {
        'story_id': get_request_arg(context, 'story_id', is_required=True),
    }

    response = get_proxy_response(context, **data)
    return ProxyResponse(response.result, response.raw_response)


@view('update_story', [Grant('stories')])
def update_story_view(context):
    required_args = ['story_id', 'object_type', 'object_id']
    data = {
        'story_id': get_request_arg(context, 'story_id', is_required=True),
    }
    for key in ['object_type', 'object_id']:
        data[key] = get_request_arg(context, key, is_required=True, from_form=True)

    data.update(get_request_args(context, from_form=True, exclude=required_args))

    response = get_proxy_response(context, **data)
    return ProxyResponse(response.result, response.raw_response)


@view('delete_story', [Grant('stories')])
def delete_story_view(context):
    data = {
        'story_id': get_request_arg(context, 'story_id', is_required=True),
    }

    response = get_proxy_response(context, **data)
    return ProxyResponse(response.result, response.raw_response)


@view('create_story_object', [Grant('stories')])
def create_story_object_view(context):
    data = {}
    required_args = ['object_type', 'object_data']
    for key in required_args:
        data[key] = get_request_arg(context, key, is_required=True, from_form=True)

    data.update(get_request_args(context, from_form=True, exclude=required_args))

    response = get_proxy_response(context, **data)
    return ProxyResponse(response.result, response.raw_response)


@view('get_story_object', [Grant('stories')])
def get_story_object_view(context):
    object_id = get_request_arg(context, 'object_id', is_required=True)
    data = {
        'object_id': object_id,
    }

    response = get_proxy_response(context, **data)
    return ProxyResponse(response.result, response.raw_response)


@view('delete_story_object', [Grant('stories')])
def delete_story_object_view(context):
    object_id = get_request_arg(context, 'object_id', is_required=True)
    data = {
        'object_id': object_id,
    }

    response = get_proxy_response(context, **data)
    return ProxyResponse(response.result, response.raw_response)


@view('get_photos', [Grant('photos')])
def photos_view(context):
    aid = get_request_arg(context, 'aid')
    next_token = get_request_arg(context, 'next_token')
    debug = get_bool_request_arg(context, 'debug_photos_pagination')

    proxy = get_proxy(context['provider']['code'], access_token=None, app=None)

    kwargs = {'next_token': next_token, 'debug': debug}
    if hasattr(proxy, 'PHOTO_ALBUM_MAPPING'):
        if not aid:
            raise InvalidParametersError('`aid` is a required parameter')
        if aid == 'personal' and context['provider']['code'] not in ['vk', 'ok', 'mr']:
            raise InvalidParametersError('`aid=personal` is not supported for this provider')
        kwargs['aid'] = aid

    try:
        response, raw_response = get_proxy_response(context, **kwargs)
    except AlbumNotExistsProxylibError:
        raw_response = None
        if aid in ['personal', 'wall']:
            # https://jira.yandex-team.ru/browse/CHEMODAN-17671
            response = {'result': []}
        else:
            raise

    logger.debug('Response next_token: %s', response.get('next_token'))
    return MultiResult(
        result=response['result'],
        additional_fields={
            'next_token': response.get('next_token'),
            'raw_response': raw_response,
        })


@view('get_user_photos', [Grant('photos')])
def user_photos_view(context):
    next_token = get_request_arg(context, 'next_token')

    try:
        response, raw_response = get_proxy_response(context, next_token=next_token)
    except AlbumNotExistsProxylibError:
        # https://jira.yandex-team.ru/browse/CHEMODAN-17671
        response = {'result': []}
        raw_response = None
    return MultiResult(
        result=response['result'],
        additional_fields={
            'next_token': response.get('next_token'),
            'raw_response': raw_response,
        })


@view('get_photo_albums', [Grant('photos')])
def photo_albums_view(context):
    locale = get_request_arg(context, 'locale')
    return get_proxy_response(context, locale=locale)


privacy_types = ['private', 'public', 'friends']


@view('create_photo_album', [Grant('photo_post')])
def photo_album_create_view(context):
    title = get_request_arg(context, 'title', is_required=True, from_form=True)
    description = get_request_arg(context, 'description', is_required=False, from_form=True)
    privacy = get_request_arg(context, 'privacy', is_required=False, from_form=True)

    if privacy and privacy not in privacy_types:
        raise InvalidParametersError('`privacy` can be one of %s' % str(privacy_types))
    return get_proxy_response(context, title=title, description=description, privacy=privacy)


@view('photo_post_get_request', [Grant('photo_post')])
def photo_post_get_request_view(context):
    aid = get_request_arg(context, 'aid', is_required=True)
    caption = get_request_arg(context, 'caption')

    return get_proxy_response(context, aid=aid, caption=caption)


@view('photo_post_commit', [Grant('photo_post')])
def photo_post_commit_view(context):
    upload_response = get_request_arg(context, 'upload_response', is_required=True, from_form=True)
    aid = get_request_arg(context, 'aid', is_required=True, from_form=True)
    caption = get_request_arg(context, 'caption', from_form=True)

    return get_proxy_response(context, upload_response=upload_response, caption=caption, aid=aid)


@view('get_videos', [Grant('videos')])
def videos_view(context):
    return get_proxy_response(context)


@view('get_auth_headers', [Grant('job-authorization')])
def sign_request(context):
    url = get_request_arg(context, 'url', is_required=True, from_form=False)
    retpath = get_request_arg(context, 'retpath', is_required=True, from_form=False)

    oauth_params = get_proxy_response(
        context,
        method=context['request'].method,
        url=url,
        post_data=context['request'].form,
    ).result

    oauth_string = 'Authorization=%s' % urllib.quote(oauth_params['Authorization'], '')
    response_header = '%s%s/url=%s' % (retpath, oauth_string, urllib.quote(url, ''))

    return Response(None, status=307, headers={'X-Accel-Redirect': response_header})


@view('refresh_token', [Grant('refresh_token')])
def refresh_token_view(context):
    return get_proxy_response(
        context,
        refresh_tokens_enabled=False,
        refresh_token=get_request_arg(context, 'refresh_token', is_required=True, from_form=True),
    )
