# -*- coding: utf-8 -*-

from __future__ import unicode_literals

import logging

from passport.backend.social.common import validators
from passport.backend.social.common.gpt import (
    Gpt,
    GptParseError,
)
from passport.backend.social.common.provider_settings import providers
from passport.backend.social.proxy2.misc import get_profile
from passport.backend.social.proxy2.views.v2.base import BaseProxyExternalHandler
from passport.backend.social.proxylib import get_proxy


logger = logging.getLogger(__name__)


class RedirectToProfileForm(validators.Schema):
    target = validators.Gpt()


class RedirectToProfile(BaseProxyExternalHandler):
    basic_form = RedirectToProfileForm

    def _process_request(self):
        gpt = Gpt()
        try:
            public, profile_id, signature = gpt.parse(self.form_values['target'])
        except GptParseError:
            self._build_unable_to_build_profile_response()
            return

        profile_row = get_profile(profile_id)
        if not profile_row:
            self._build_unable_to_build_profile_response()
            return

        true_signature = gpt.calc_signature(public, profile_row.uid)

        if true_signature != signature:
            self._build_unable_to_build_profile_response()
            return

        profile_links = self._get_profile_links(profile_row)

        if not profile_links:
            self._build_unable_to_build_profile_response()
            return

        logger.debug('Redirect to profile link: %s' % profile_links[0])
        self._build_redirect_response(profile_links[0])

    def _get_profile_links(self, profile_row):
        provider_info = providers.get_provider_info_by_id(profile_row.provider_id)
        if not provider_info:
            return list()
        proxy = get_proxy(code=provider_info['code'])
        return proxy.get_profile_links(
            userid=profile_row.userid,
            username=profile_row.username,
        )

    def _build_redirect_response(self, redirect_url):
        response = self._response_class(
            status=307,
            headers={'Location': redirect_url},
        )
        self.response_values.update({'response': response})

    def _build_unable_to_build_profile_response(self):
        return self._build_plain_text_response('Unable to build profile link')

    def _build_plain_text_response(self, message):
        response = self._response_class(status=200, mimetype='text/plain')
        response.data = message + '\n'
        self.response_values.update({'response': response})

    def _response_success(self):
        response = self.response_values['response']
        response.api_status = 'ok'
        return response
