# -*- coding: utf-8 -*-
"""
Identity token

  * Наполнение

    https://developer.apple.com/documentation/signinwithapplerestapi/authenticating_users_with_sign_in_with_apple#3383773

  * Проверка

    https://developer.apple.com/documentation/signinwithapplerestapi/verifying_a_user#3383769
"""
from __future__ import unicode_literals

import datetime
import json
import logging

from jwt import PyJWS
import jwt.algorithms
import jwt.exceptions
from passport.backend.social.common import validators
from passport.backend.social.common.chrono import now
from passport.backend.social.common.exception import (
    InvalidTokenProxylibError,
    UnexpectedResponseProxylibError,
)
from passport.backend.social.common.misc import (
    dump_to_json_string,
    trim_message,
)
from passport.backend.social.common.providers.Apple import Apple
from passport.backend.social.common.social_config import social_config
from passport.backend.social.proxylib import mapper
from passport.backend.social.proxylib.proxy import (
    parse_access_token_from_authorization_code_response,
    SocialPackedTokenParser,
    SocialPackedTokenVersionForm,
    SocialProxy,
)
from passport.backend.utils.common import noneless_dict


logger = logging.getLogger(__name__)

APPLE_PUBLIC_KEYS_URL = 'https://appleid.apple.com/auth/keys'
APPLE_REVOKE_TOKEN_URL = 'https://appleid.apple.com/auth/revoke'
APPLE_TOKEN_ISSUER = 'https://appleid.apple.com'


def parse_apple_client_token(doc):
    """
    Парсит упакованный токен от АМа из строки в объект
    """
    parser = SocialPackedTokenParser(_APPLE_SOCIAL_PACKED_TOKEN_VERSION_FORMS)
    token_dict = parser.parse(doc)
    return AppleClientToken(**token_dict)


def get_apple_jwt_certificate(certificate_id):
    prefix = 'apple_jwt_certificate_'
    attr_name = prefix + certificate_id
    if not hasattr(social_config, attr_name):
        logger.error('social_config.%s not found, using default' % attr_name)
        attr_name = prefix + social_config.apple_jwt_certificate_id
    return getattr(social_config, attr_name)


class AppleProxy(SocialProxy):
    code = Apple.code

    def get_profile(self):
        token_info = self.get_token_info()
        profile = dict(userid=token_info['userid'])
        if (
            token_info.get('email') and
            token_info.get('email_verified') and
            # Apple не возвращает is_private_email, когда e-mail настоящий
            not token_info.get('is_private_email')
        ):
            email = token_info['email']
            if not email.endswith('@privaterelay.appleid.com'):
                profile.update(email=email)
        return profile

    def get_token_info(self, need_client_id=True):
        raw_id_token = self.r.access_token['value']
        id_token = self._check_id_token(raw_id_token)

        try:
            id_token = AppleIdTokenForm().to_python(id_token)
        except validators.Invalid as e:
            key = e.error_dict.keys()[0]
            raise UnexpectedResponseProxylibError('Invalid %s: %s' % (key, raw_id_token))

        token_info = dict(
            client_id=id_token['aud'],
            email=id_token['email'],
            email_verified=id_token['email_verified'],
            expires=id_token['exp'],
            is_private_email=id_token['is_private_email'],
            userid=id_token['sub'],
        )
        return noneless_dict(token_info)

    def exchange_token(self):
        apple_client_token = parse_apple_client_token(self.r.access_token['value'])
        self._check_id_token(apple_client_token.id_token)
        return dict(value=apple_client_token.id_token)

    def client_token_to_profile(self, client_token):
        apple_client_token = parse_apple_client_token(client_token.value)
        return dict(
            firstname=apple_client_token.firstname,
            lastname=apple_client_token.lastname,
        )

    def get_public_keys(self):
        self.r.compose_request(
            base_url=APPLE_PUBLIC_KEYS_URL,
            add_access_token=False,
        )
        self.r.execute_request_basic()
        self.r.deserialize_json()

        try:
            return self.r.context['data']['keys']
        except (KeyError, TypeError):
            logger.debug('Failed to get public keys: %s' % trim_message(self.r.context['raw_response']))
            raise UnexpectedResponseProxylibError('Failed to get public keys')

    def revoke_token(self):
        self.r.compose_request(
            add_access_token=False,
            base_url=APPLE_REVOKE_TOKEN_URL,
            data=dict(
                client_id=self.r.app.id,
                client_secret=self.get_client_secret(),
                token=self.r.access_token['value'],
                token_type_hint='access_token',
            ),
        )
        self.r.execute_request_basic()

        if self.r.context['raw_response'].status_code // 100 != 2:
            try:
                parse_access_token_from_authorization_code_response(self.r.context['raw_response'].content)
            except InvalidTokenProxylibError:
                pass

    def _check_id_token(self, raw_id_token):
        """
        Проверяет что токен
          * Создан системой Apple ID
          * Токен действует

        Не проверяет что токен выдан Яндексовому приложения, потому что
          * Дорогая проверка требующая похода в БД
          * Такая проверка выполняется AppleCommunicator'ом в момент попадания
            токена в контур
        """
        jws = PyJWS()

        # Определяем идентификатор ключа, которым подписан id_token
        try:
            id_token_header = jws.get_unverified_header(raw_id_token)
        except jwt.exceptions.InvalidTokenError:
            raise InvalidTokenProxylibError('Token header corrupted: %s' % raw_id_token)
        kid = id_token_header.get('kid')

        pubkeys = self.get_public_keys()

        # Ищем ключ, которым подписан id_token
        for pubkey in pubkeys:
            if pubkey['kid'] == kid:
                break
        else:
            raise InvalidTokenProxylibError('Unknown signing key: %s' % raw_id_token)

        # Проверка, что токен подписан безопасным алгоритмом
        if pubkey['alg'] not in social_config.allowed_json_web_algorithms:
            logger.error('Apple provides JWK with forbidden alg: %s' % pubkey['alg'])
            raise InvalidTokenProxylibError('Forbidden Public Key JSON Web Algorithm: %s' % pubkey['alg'])

        # Проверка подписи и декодирование
        algs = jwt.algorithms.get_default_algorithms()
        pubkey_alg = algs.get(pubkey['alg'])
        allowed_algs = {pubkey['alg']: pubkey_alg}
        pubkey = pubkey_alg.from_jwk(dump_to_json_string(pubkey))
        try:
            id_token = jws.decode(raw_id_token, key=pubkey, algorithms=allowed_algs)
        except jwt.exceptions.InvalidTokenError:
            raise InvalidTokenProxylibError('Token corrupted: %s' % raw_id_token)

        try:
            id_token = json.loads(id_token)
        except ValueError:
            raise InvalidTokenProxylibError('Invalid JSON: %s' % raw_id_token)

        if not isinstance(id_token, dict):
            raise InvalidTokenProxylibError('Token is not object: %s' % raw_id_token)

        if id_token.get('iss') != APPLE_TOKEN_ISSUER:
            raise InvalidTokenProxylibError('Token iss should be %s: %s' % (APPLE_TOKEN_ISSUER, raw_id_token))

        try:
            exp = int(id_token.get('exp'))
        except (TypeError, ValueError):
            exp = 0
        if exp <= now.f():
            raise InvalidTokenProxylibError('Token expired: %s' % raw_id_token)

        return id_token

    def get_client_secret(self):
        iat = now.i()
        jwt_certificate_id = self.r.app.apple_jwt_certificate_id or social_config.apple_jwt_certificate_id
        return jwt.encode(
            payload=dict(
                iss=self.r.app.apple_team_id or social_config.apple_team_id,
                iat=iat,
                exp=iat + int(datetime.timedelta(hours=1).total_seconds()),
                aud=APPLE_TOKEN_ISSUER,
                sub=self.r.app.id,
            ),
            key=get_apple_jwt_certificate(jwt_certificate_id),
            headers=dict(
                alg='ES256',
                kid=jwt_certificate_id,
            ),
        )


class AppleClientToken(object):
    """
    Упакованный токен из АМа
    """
    def __init__(
        self,
        authorization_code=None,
        id_token=None,
        version=None,
        firstname=None,
        lastname=None,
    ):
        self.authorization_code = authorization_code
        self.id_token = id_token
        self.version = version
        self.firstname = firstname
        self.lastname = lastname


class AppleIdTokenForm(validators.Schema):
    """
    Форма для Identity token
    """
    aud = validators.String()
    exp = validators.Int()
    sub = validators.String()
    email = validators.String(if_missing=None, not_empty=False)
    email_verified = validators.StringBool(if_missing=None)
    is_private_email = validators.StringBool(if_missing=None)


class AppleClientTokenV1Form(SocialPackedTokenVersionForm):
    """
    Форма для упакованного токена из АМа
    """
    authorization_code = validators.String(not_empty=True)
    id_token = validators.String(not_empty=True)
    firstname = validators.String(if_empty=None, if_missing=None)
    lastname = validators.String(if_empty=None, if_missing=None)


_APPLE_SOCIAL_PACKED_TOKEN_VERSION_FORMS = {1: AppleClientTokenV1Form}


mapper.add_mapping(AppleProxy.code, AppleProxy)
