# -*- coding: utf-8 -*-

from __future__ import unicode_literals

import logging

from passport.backend.social.common import oauth2
from passport.backend.social.common.misc import trim_message
from passport.backend.social.common.providers.DnevnikRu import DnevnikRu
from passport.backend.social.common.social_config import social_config
from passport.backend.social.proxylib import mapper
from passport.backend.social.proxylib.proxy import (
    RefreshTokenGetter,
    SocialProxy,
)


logger = logging.getLogger(__name__)


class DnevnikRuRefreshTokenGetter(RefreshTokenGetter):
    RESULT_TRANSLATION = {
        'expiresIn': 'expires_in',
        'accessToken': 'access_token',
        'refreshToken': 'refresh_token',
    }

    def _build_request_data(self):
        return {
            'grant_type': 'RefreshToken',
            'RefreshToken': self.refresh_token,
        }

    def _parse_json(self):
        super(DnevnikRuRefreshTokenGetter, self)._parse_json()
        decoded = self.r.context['data']
        for _from, _to in self.RESULT_TRANSLATION.items():
            if _from in decoded:
                decoded[_to] = decoded[_from]


class DnevnikRuProxy(SocialProxy):
    code = DnevnikRu.code
    REFRESH_TOKEN_GETTER_CLASS = DnevnikRuRefreshTokenGetter

    @property
    def SETTINGS(self):
        return {
            'get_profile_url': social_config.dnevnik_ru_get_profile_url,
            'error_codes_invalid_token': {401},
            'oauth_refresh_token_url': social_config.dnevnik_ru_refresh_token_url,
            'oauth_auth_over_header': True,
        }

    PROFILE_MAPPING = {
        'id': 'userid',
        'firstName': 'firstname',
        'lastName': 'lastname',
        'shortName': '_shortname',   # используем в convert как firstname,
                                     # если нет firstName, lastName
    }

    def convert_profile(self, profile):
        shortname = profile.pop('_shortname', None)
        has_name = bool(profile.get('firstname') or profile.get('lastname'))
        if not has_name and shortname:
            profile['firstname'] = shortname

    def get_profile(self):
        self.r.compose_request(
            url_name='get_profile_url',
            additional_headers={'Access-Token': self.r.access_token['value']},
        )
        self.r.execute_request_basic(http_method='GET')
        logger.debug(
            'Profile response %s' % trim_message(self.r.context['raw_response'].decoded_data),
        )
        self.r.parse_error_response()
        self.r.deserialize_json()
        self.r.extract_response_data(
            self.PROFILE_MAPPING,
            converter=self.convert_profile,
            listed=False,
            one=True,
        )
        return self.r.context['processed_data']

    def refresh_token(self, refresh_token):
        return self._refresh_token(
            refresh_token, self._detect_refresh_token_error, auth_type_is_basic=True,
        )

    def _detect_refresh_token_error(self, response):
        try:
            oauth2.refresh_token.detect_error(response)
        except oauth2.refresh_token.InvalidRequest as e:
            raise oauth2.refresh_token.InvalidGrant.from_exception(e)


mapper.add_mapping(DnevnikRu.code, DnevnikRuProxy)
