# -*- coding: utf-8 -*-

from __future__ import unicode_literals

import copy
import datetime
import json

from cryptography import x509
from cryptography.hazmat.primitives import serialization
from cryptography.hazmat.primitives.serialization import pkcs7
import jwt
import jwt.exceptions
from passport.backend.social.common import (
    crypto,
    exception as exceptions,
    validators,
)
from passport.backend.social.common.chrono import now
from passport.backend.social.common.misc import (
    dump_to_json_string,
    split_scope_string,
)
from passport.backend.social.common.providers.Esia import Esia
from passport.backend.social.common.social_config import social_config
from passport.backend.social.proxylib.proxy import (
    RefreshTokenGetter,
    SocialPackedTokenParser,
    SocialPackedTokenSerializer,
    SocialPackedTokenVersionForm,
    SocialProxy,
)
from passport.backend.utils.common import (
    from_base64_url,
    to_base64_url,
)
import phonenumbers
import pytz

from . import mapper


def parse_esia_token(doc):
    parser = SocialPackedTokenParser(_ESIA_TOKEN_VERSIONS, compression=True)
    token_dict = parser.parse(doc)
    return EsiaToken(**token_dict)


def parse_esia_refresh_token(doc):
    parser = SocialPackedTokenParser(_ESIA_REFRESH_TOKEN_VERSIONS, compression=True)
    token_dict = parser.parse(doc)
    return EsiaRefreshToken(**token_dict)


def get_esia_timestamp():
    return now(pytz.timezone(social_config.time_zone)).strftime('%Y.%m.%d %H:%M:%S %z')


def get_esia_signature(params):
    signature = pkcs7.PKCS7SignatureBuilder().set_data(
        data=get_esia_request_id(params),
    ).add_signer(
        certificate=get_esia_yandex_certificate(),
        private_key=get_esia_yandex_private_key(),
        hash_algorithm=crypto.GostR3411_2012_256(),
    ).sign(
        backend=crypto.get_social_cryptography_backend(),
        encoding=serialization.Encoding.DER,
        options=[
            pkcs7.PKCS7Options.Binary,
            pkcs7.PKCS7Options.DetachedSignature,
            pkcs7.PKCS7Options.NoAttributes,
        ],
    )

    return to_base64_url(signature)


def verify_esia_signature(params, signature):
    if isinstance(signature, unicode):
        signature = signature.encode('utf-8')
    certs = crypto.Certificates()
    certs.push(get_esia_yandex_certificate())
    crypto.pkcs7_verify(
        certificates=certs,
        data=get_esia_request_id(params),
        signature=from_base64_url(signature),
    )


def get_esia_request_id(params):
    bits = [params.get(p, '') for p in ['scope', 'timestamp', 'client_id', 'state']]
    return ''.join(bits).encode('utf-8')


def get_esia_yandex_certificate():
    return x509.load_pem_x509_certificate(social_config.esia_yandex_certificate, crypto.get_social_cryptography_backend())


def get_esia_yandex_private_key():
    return serialization.load_pem_private_key(
        social_config.esia_yandex_private_key,
        social_config.esia_yandex_private_key_password,
        crypto.get_social_cryptography_backend(),
    )


def get_esia_jwt_certificate():
    return x509.load_pem_x509_certificate(
        social_config.esia_jwt_certificate,
        crypto.get_social_cryptography_backend(),
    )


def parse_and_verify_esia_id_token(
    raw_id_token,
    verify_aud=True,
    verify_exp=True,
    verify_iss=True,
    verify_nbf=True,
    verify_sbt=True,
    verify_signature=True,
    verify_subject_type=True,
):
    try:
        id_token_header = jwt.get_unverified_header(raw_id_token)
    except jwt.exceptions.InvalidTokenError:
        raise exceptions.InvalidTokenProxylibError('Id token header corrupted: %s' % raw_id_token)

    expected_sbt = 'id'
    if verify_sbt and id_token_header.get('sbt') != expected_sbt:
        raise exceptions.InvalidTokenProxylibError(
            'Id token header sbt should be "%s": %s' % (expected_sbt, id_token_header.get('sbt')),
        )

    expected_alg = 'GOST3410_2012_256'
    if verify_signature and id_token_header.get('alg') != expected_alg:
        raise exceptions.InvalidTokenProxylibError(
            'Id token header alg should be "%s": %s' % (expected_alg, id_token_header.get('alg')),
        )

    decode_opts = dict(
        verify_signature=False,
        verify_aud=False,
        verify_exp=False,
        verify_iat=False,
        verify_iss=False,
        verify_nbf=False,
    )
    kwargs = dict()

    if verify_iss:
        decode_opts.update(verify_iss=True)
        kwargs.update(issuer='http://%s/' % social_config.esia_real_host)

    if verify_aud:
        decode_opts.update(verify_aud=True)
        kwargs.update(audience='YANDEXID')

    if verify_nbf:
        decode_opts.update(verify_nbf=True)
        # Leeway -- максимальное допустимое отставание местного времени
        kwargs.update(leeway=datetime.timedelta(hours=1))

    if verify_signature:
        decode_opts.update(verify_signature=True)
        kwargs.update(key=get_esia_jwt_certificate().public_key())

    if verify_exp:
        decode_opts.update(verify_exp=True)

    try:
        id_token = jwt.decode(raw_id_token, options=decode_opts, **kwargs)
    except jwt.exceptions.InvalidTokenError as e:
        raise exceptions.InvalidTokenProxylibError('Id token corrupted: %s, %s' % (str(e), raw_id_token))

    if verify_subject_type:
        subject_type = id_token.get('urn:esia:sbj', dict()).get('urn:esia:sbj:typ')
        if not (subject_type and subject_type.upper() == 'P'):
            raise exceptions.InvalidTokenProxylibError('Invalid Id token subject type: %s' % subject_type)

    return id_token


class EsiaRefreshTokenGetter(RefreshTokenGetter):
    def _build_request(self):
        data, headers = super(EsiaRefreshTokenGetter, self)._build_request()
        esia_token = parse_esia_refresh_token(self.refresh_token)
        data = copy.copy(data)
        data.update(
            refresh_token=esia_token.value,
            scope=esia_token.scope,
            state=esia_token.state,
            timestamp=get_esia_timestamp(),
        )
        data['client_secret'] = get_esia_signature(data)
        return data, headers

    def _parse_result(self):
        super(EsiaRefreshTokenGetter, self)._parse_result()

        data = self.r.context['processed_data']
        esia_token = parse_esia_refresh_token(self.refresh_token)

        data['access_token'] = EsiaToken(
            access_token=data.get('access_token', ''),
            id_token=esia_token.id_token,
            scope=esia_token.scope,
            version=1,
        ).serialize()

        if data.get('refresh_token'):
            data['refresh_token'] = EsiaRefreshToken(
                id_token=esia_token.id_token,
                scope=esia_token.scope,
                state=esia_token.state,
                value=data['refresh_token'],
                version=1,
            ).serialize()

        return data


class EsiaProxy(SocialProxy):
    code = Esia.code
    profile_mapping = {
        'birthDate': 'birthday',
        'trusted': 'is_verified',
        'firstName': 'firstname',
        'lastName': 'lastname',
        'middleName': 'middlename',
        'userid': 'userid',
    }
    REFRESH_TOKEN_GETTER_CLASS = EsiaRefreshTokenGetter

    @property
    def SETTINGS(self):
        return dict(
            birthday_regexp=r'^(?P<day>\d{1,2})\.(?P<month>\d{1,2})\.?(?P<year>\d{4})?$',
            error_codes_invalid_token={'ESIA-005011'},
            esia_host=social_config.esia_host,
            esia_real_host=social_config.esia_real_host,
            oauth_auth_over_header=True,
            oauth_refresh_token_url=social_config.esia_token_url,
        )

    def __init__(self):
        super(EsiaProxy, self).__init__()
        self.esia_token = None

    def get_profile(self):
        scopes = set(split_scope_string(self.r.access_token.get('scope', '')))
        if (scopes - set(['openid'])):
            esia_request = EsiaRequest()
            esia_request.schema('/rs/model/prn/Person-3')
            esia_request.method('/rs/prns/%s' % self._userid)

            if any(c in scopes for c in ['email', 'mobile']):
                esia_request.embed('contacts.elements-1')

            if 'id_doc' in scopes:
                esia_request.embed('documents.elements-1')

            esia_request.compose(self)

            self.r.execute_request_basic()
            self.r.deserialize_json()
            self._parse_error_response()

            self._check_social_profile_enabled()
        else:
            self.r.context['data'] = dict()

        self.r.context['data'].update(userid=self._userid)

        self.r.extract_response_data(
            converter=self._parse_profile,
            listed=False,
            mapping=self.profile_mapping,
            one=True,
        )

        return self.r.context['processed_data']

    def refresh_token(self, refresh_token):
        return self._refresh_token(refresh_token)

    def _parse_error_response(self):
        response = self.r.context.get('data')
        if not isinstance(response, dict):
            raise exceptions.UnexpectedResponseProxylibError()

        if response.get('code'):
            self.r.raise_correct_exception(response['code'], response.get('message'))

    def _check_social_profile_enabled(self):
        if self.r.context.get('data', dict()).get('status') != 'REGISTERED':
            raise exceptions.SocialUserDisabledProxylibError()

    def _parse_profile(self, profile):
        self.r.convert_birthday(profile)
        profile.update(
            email=self._get_email(),
            phone=self._get_phone(),
        )

    def _get_email(self):
        for contact in self._get_contacts():
            if (
                contact and
                isinstance(contact, dict) and
                contact.get('type') == 'EML' and
                contact.get('vrfStu') == 'VERIFIED' and
                contact.get('value')
            ):
                return contact['value']

    def _get_phone(self):
        for contact in self._get_contacts():
            if (
                contact and
                isinstance(contact, dict) and
                contact.get('type') == 'MBT' and
                contact.get('vrfStu') == 'VERIFIED' and
                contact.get('value')
            ):
                phone_number = contact['value']
                phone_number = phonenumbers.parse(phone_number, 'RU')
                if phonenumbers.is_valid_number(phone_number):
                    return phonenumbers.format_number(phone_number, phonenumbers.PhoneNumberFormat.E164)

    def _get_contacts(self):
        data = self.r.context.get('data')

        if not (data and isinstance(data, dict)):
            return list()
        contacts = data.get('contacts')

        if not (contacts and isinstance(contacts, dict)):
            return list()
        elements = contacts.get('elements')

        if not (elements and isinstance(elements, list)):
            return list()

        return elements

    @property
    def _userid(self):
        raw_id_token = self.r.access_token.get('id_token', '')
        # Предполагаю, что сюда id_token будет попадать только из доверенного
        # контура, поэтому пропускаю большую часть проверок.
        # Кроме того после обновления токена, его id_token обновляется, а
        # потому его нужно проверять только в момент первого входа в довернный
        # контур. Иначе get_profile не сможет работаеть, потому что userid
        # храниться в id_token и другими ручками Есиа не отдаётся.
        id_token = parse_and_verify_esia_id_token(
            raw_id_token,
            verify_aud=True,
            verify_exp=False,
            verify_iss=True,
            verify_nbf=False,
            verify_sbt=True,
            verify_signature=False,
            verify_subject_type=True,
        )
        return str(id_token.get('sub'))

    def _schema(self, path):
        return 'https://%(host)s%(path)s' % dict(
            host=self.SETTINGS.get('esia_real_host', ''),
            path=path,
        )

    def _url(self, path):
        return 'https://%(host)s%(path)s' % dict(
            host=self.SETTINGS.get('esia_host', ''),
            path=path,
        )

    def parse_raw_token(self, raw_token):
        super(EsiaProxy, self).parse_raw_token(raw_token)

        self.esia_token = parse_esia_token(self.r.access_token.get('value', ''))
        self.r.access_token = copy.copy(self.r.access_token)
        self.r.access_token.update(
            id_token=self.esia_token.id_token,
            scope=self.esia_token.scope,
            value=self.esia_token.access_token,
        )


class EsiaToken(object):
    def __init__(
        self,
        access_token=None,
        id_token=None,
        scope=None,
        version=None,
    ):
        self.access_token = access_token
        self.id_token = id_token
        self.scope = scope
        self.version = version

    def serialize(self):
        return SocialPackedTokenSerializer(_ESIA_TOKEN_VERSIONS, compression=True).serialize(self.to_dict())

    def to_dict(self):
        return dict(
            access_token=self.access_token,
            id_token=self.id_token,
            scope=self.scope,
            version=self.version,
        )


class EsiaTokenV1Form(SocialPackedTokenVersionForm):
    access_token = validators.String(not_empty=True)
    id_token = validators.String(not_empty=True)
    scope = validators.ScopeString()


_ESIA_TOKEN_VERSIONS = {1: EsiaTokenV1Form}


class EsiaRefreshToken(object):
    def __init__(
        self,
        id_token=None,
        scope=None,
        state=None,
        value=None,
        version=None,
    ):
        self.id_token = id_token
        self.scope = scope
        self.state = state
        self.value = value
        self.version = version

    def serialize(self):
        return SocialPackedTokenSerializer(_ESIA_REFRESH_TOKEN_VERSIONS, compression=True).serialize(self.to_dict())

    def to_dict(self):
        return dict(
            id_token=self.id_token,
            scope=self.scope,
            state=self.state,
            value=self.value,
            version=self.version,
        )


class EsiaRefreshTokenV1Form(SocialPackedTokenVersionForm):
    id_token = validators.String(not_empty=True)
    scope = validators.ScopeString()
    state = validators.String(not_empty=True)
    value = validators.String(not_empty=True)


_ESIA_REFRESH_TOKEN_VERSIONS = {1: EsiaRefreshTokenV1Form}


class EsiaRequest(object):
    def __init__(self):
        self._embeds = set()
        self._method = None
        self._schema = None

    def compose(self, esia_proxy):
        additional_args = dict()
        if self._embeds:
            additional_args.update(embed=self.format_embed())

        additional_headers = dict(accept=self.format_accept(esia_proxy))

        esia_proxy.r.compose_request(
            additional_args=additional_args,
            additional_headers=additional_headers,
            base_url=esia_proxy._url(self._method),
        )

    def embed(self, value):
        self._embeds.add(value)

    def format_accept(self, esia_proxy):
        bits = ['application/json']
        if self._schema:
            bits.append('schema="%s"' % esia_proxy._schema(self._schema))
        return '; '.join(bits)

    def format_embed(self):
        return '(%s)' % ','.join(sorted(self._embeds))

    def method(self, value):
        self._method = value

    def schema(self, value):
        self._schema = value


class EsiaPermissions(object):
    def __init__(self):
        self._perms = None

    @classmethod
    def from_params(cls, params):
        self = cls()

        scopes = split_scope_string(params.get('scope'))
        scopes = [s for s in scopes if s != 'openid']
        if not scopes:
            return

        self._perms = [
            dict(
                # Тип согласия
                sysname='YANDEX_NEW_USER',

                # Цели согласия
                purposes=[
                    dict(sysname='IDENTIFICATION'),
                ],

                # Действия с данными
                actions=[
                    dict(sysname='ALL_ACTIONS_TO_DATA'),
                ],

                scopes=[dict(sysname=s) for s in scopes],
            ),
        ]

        return self

    @classmethod
    def from_list(cls, permissions):
        self = cls()
        self._perms = permissions
        return self

    @classmethod
    def from_str(cls, data):
        return cls.from_list(json.loads(from_base64_url(data)))

    def to_list(self):
        return list(self._perms) if self._perms else list()

    def __str__(self):
        return to_base64_url(dump_to_json_string(self._perms, minimal=True).encode('utf-8'))


mapper.add_mapping(EsiaProxy.code, EsiaProxy)
