# -*- coding: utf-8 -*-

from __future__ import unicode_literals

from . import mapper
from .proxy import (
    paginated,
    parse_access_token_from_authorization_code_response,
    SocialProxy,
)


class FacebookProxy(SocialProxy):
    code = 'fb'

    SETTINGS = {
        'birthday_regexp': r'^(?P<month>\d{2})/(?P<day>\d{2})/?(?P<year>\d{4})?$',
        'gender_map': {'male': 'm', 'female': 'f'},
        'graph_url': 'https://graph.facebook.com/v10.0/%(method)s',
        'oauth_access_token_url': 'https://graph.facebook.com/v10.0/oauth/access_token',

        'error_codes_invalid_token': [190],
        'error_codes_permission': range(200, 300),
        'error_codes_service_unavailable': [1, 2],
    }

    PROFILE_MAPPING = {
        'id': 'userid',
        'first_name': 'firstname',
        'last_name': 'lastname',
        'middle_name': 'middlename',
        'gender': 'gender',
        'birthday': 'birthday',
        'email': 'email',
        'picture': 'avatar',
        'friends': 'friends',
        'token_for_business': 'token_for_business',
    }

    TOKEN_MAPPING = {
        'expires_at': 'expires',
        'is_valid': 'valid',
        'issued_at': 'issued',
        'scopes': 'scopes',
        'user_id': 'userid',
        'app_id': 'client_id',
    }

    def get_app_token(self):
        # Для FB получать токен для приложения не обязательно: можно использовать пару client_id|client_secret
        return '%s|%s' % (self.r.app.id, self.r.app.secret)

    def get_profile(self):
        fields = self.PROFILE_MAPPING.keys()
        fields.remove('picture')
        fields.append('picture.type(large)')
        fields.remove('friends')
        fields.append('friends.limit(0)')

        self.r.compose_request(url_name='graph_url', method='me', fields=fields)
        self.r.execute_request_basic()
        self.r.deserialize_json()
        self.r.parse_error_response()

        self.r.extract_response_data(self.PROFILE_MAPPING, one=True, converter=self.r.convert_profile, listed=False)
        self.r.convert_birthday()
        self.r.convert_gender()

        return self.r.context['processed_data']

    @paginated
    def get_friends(self):
        """
        https://developers.facebook.com/docs/graph-api/reference/v10.0/user/friends

        Нужно чтобы разрешение :scope:`user_friends` было выдано и пользователем
        и его другом.
        """
        fields = self.PROFILE_MAPPING.keys()
        fields.remove('picture')
        fields.append('picture.type(large)')
        fields.remove('friends')
        fields.remove('token_for_business')

        friends_per_request = 100
        offset = 0

        while True:
            args = {
                'offset': offset,
                'limit': friends_per_request,
            }
            self.r.compose_request(url_name='graph_url', method='me/friends', fields=fields, additional_args=args)

            self.r.execute_request_basic()
            self.r.deserialize_json()
            self.r.parse_error_response()
            self.r.extract_response_data(
                self.PROFILE_MAPPING,
                one=False,
                converter=self.r.convert_profile,
                extract_field='data',
            )
            self.r.convert_friends_gender()
            self.r.convert_friends_birthday()
            offset += len(self.r.context['processed_data'])
            yield self.r.context['processed_data']

            if len(self.r.context['processed_data']) < friends_per_request:
                break

    def get_token_info(self, need_client_id=True):
        """
        https://developers.facebook.com/docs/graph-api/reference/v10.0/debug_token
        """
        self.r.compose_request(
            url_name='graph_url',
            method='debug_token',
            add_access_token=False,
            additional_args={
                'input_token':  self.r.access_token['value'],
                'access_token': self.get_app_token(),
            },
        )
        self.r.execute_request_basic()
        self.r.deserialize_json()
        self.r.parse_error_response()
        self.r.extract_response_data(self.TOKEN_MAPPING, one=True, listed=False, converter=None, extract_field='data')
        return self.r.context['processed_data']

    def exchange_token(self):
        args = {
            'grant_type': 'fb_exchange_token',
            'client_id': self.r.app.id,
            'client_secret': self.r.app.secret,
            'fb_exchange_token': self.r.access_token['value'],
        }
        self.r.compose_request(url_name='oauth_access_token_url', additional_args=args)

        self.r.execute_request_basic(parser=self._parse_exchange_token_response)
        self._parse_exchange_token_response()

        return self.r.context['data']

    def _parse_exchange_token_response(self):
        raw_response = self.r.context['raw_response']
        self.r.context['data'] = parse_access_token_from_authorization_code_response(
            response=raw_response.decoded_data,
            detect_error=self.r.parse_error_dict,
        )


mapper.add_mapping(FacebookProxy.code, FacebookProxy)
