# -*- coding: utf-8 -*-
from __future__ import unicode_literals

from passport.backend.social.common import oauth2
from passport.backend.social.common.exception import (
    BadParametersProxylibError,
    InvalidTokenProxylibError,
    ProviderTemporaryUnavailableProxylibError,
)

from . import mapper
from .proxy import SocialProxy


class GoogleProxy(SocialProxy):
    code = 'gg'

    SETTINGS = {
        'birthday_regexp': r'^(?P<year>\d{4})\-(?P<month>\d{2})\-(?P<day>\d{2})$',
        'gender_map': {'male': 'm', 'female': 'f'},
        'api_url_oauth2_v3': 'https://www.googleapis.com/oauth2/v3/%(method)s',
        'api_url_atom_feed': 'https://mail.google.com/mail/feed/atom/',
        'api_url_picasa': 'https://picasaweb.google.com/data/feed/api/%(method)s',
        'oauth_auth_over_header': True,
        'oauth_refresh_token_url': 'https://www.googleapis.com/oauth2/v3/token',
        'oauth_access_token_url': 'https://www.googleapis.com/oauth2/v3/token',
        'treat_invalid_scope_as_invalid_token': True,

        'error_codes_permission': [403, 'invalid_grant'],
        'error_codes_invalid_token': [401, 'invalid_token'],
        'error_codes_service_unavailable': [503],
    }

    # Можно забирать произвольное количество данных за раз, но это слишком долго.
    PHOTOS_PER_REQUEST = 500
    PHOTO_REQUESTS_PER_SESSION = 1

    PRIVACY_LOCAL_BY_GLOBAL = {
        'friends': 'protected',
        'public': 'public',
        'private': 'private',
    }

    PRIVACY_GLOBAL_BY_LOCAL = {
        'protected': 'friends',
        'private': 'private',
        'public': 'public',
        'all': 'public',
        'visible': 'public',
    }

    PROFILE_MAPPING = {
        'sub': 'userid',
        'given_name': 'firstname',
        'family_name': 'lastname',
        'gender': 'gender',
        'email': 'email',
    }

    AVATAR_MAPPING = {
        'picture': 'avatar.0x0',
    }
    PROFILE_MAPPING.update(AVATAR_MAPPING)

    PHOTO_ALBUM_MAPPING = {
        'gphoto:id': ('aid', None),
        'gphoto:timestamp': ('created', lambda msec_str: int(msec_str) / 1000),
        'atom:title': ('title', None),
        'atom:summary': ('description', None),
        'gphoto:access': ('visibility', None),
        'gphoto:numphotos': ('photo_count', int),
    }

    PHOTO_MAPPING = {
        'gphoto:id': ('pid', None),
        'gphoto:width': ('width', int),
        'gphoto:height': ('height', int),
        'gphoto:timestamp': ('created', lambda msec_str: int(msec_str) / 1000),
        'atom:content/@src': ('url', None),
        'georss:where/gml:Point/gml:pos': ('location', 'get_location'),
        'exif:tags': {
            'exif:time': ('photo_creation_time', lambda msec_str: int(msec_str) / 1000),
        },
    }

    TOKEN_MAPPING = {
        'sub': 'userid',
        'scope': 'scopes',
        'exp': 'expires',
        'aud': 'client_id',
    }

    def get_profile(self):
        self.r.compose_request(url_name='api_url_oauth2_v3', method='userinfo')
        self.r.execute_request_basic()
        self.r.deserialize_json()
        self.r.parse_error_response()
        self.r.extract_response_data(self.PROFILE_MAPPING, self.r.convert_profile,  listed=False, one=True)
        self.r.convert_gender()
        return self.r.context['processed_data']

    def get_mails_unread_count(self):
        # TODO error detection
        self.r.compose_request(url_name='api_url_atom_feed')

        def parse():
            self.r.parse_error_code_value()
            self.r.deserialize_xml()
        self.r.execute_request_basic(parser=parse)
        parse()

        self.r.extract_mails_unread_count()
        return self.r.context['processed_data']

    def get_photo_albums(self):
        # required scope - https://picasaweb.google.com/data/
        fields = 'entry(gphoto:id,gphoto:timestamp,title,summary,gphoto:access,gphoto:numphotos)'

        self.r.compose_request(
            url_name='api_url_picasa',
            method='user/default',
            additional_headers={
                'GData-Version': 2,
                'deprecation-extension': 'true',
            },
            additional_args={
                'fields': fields,
                'deprecation-extension': 'true',
            }
        )

        def parse():
            self.r.parse_error_code_value()
            self.r.deserialize_xml()
        self.r.execute_request_basic(timeout=10, parser=parse)
        parse()

        self.r.extract_xml_entries(self.PHOTO_ALBUM_MAPPING, converter=self.r.convert_album)
        self.r.convert_photo_albums_privacy(self.PRIVACY_GLOBAL_BY_LOCAL)
        return self.r.context['processed_data']

    def get_photos(self, aid, next_token=None, debug=False):
        # required scope - https://picasaweb.google.com/data/

        try:
            next_token = int(str(next_token))
        except ValueError:
            next_token = 1

        args = {
            'kind': 'photo',
            'imgmax': 'd',
            'max-results': self.PHOTOS_PER_REQUEST,
            'start-index': next_token,
            'deprecation-extension': 'true',
        }

        self.r.compose_request(
            url_name='api_url_picasa',
            method='user/default/albumid/' + aid,
            additional_args=args,
            additional_headers={
                'GData-Version': 2,
                'deprecation-extension': 'true',
            },
        )

        def parse():
            self.r.parse_error_code_value()
            self.r.deserialize_xml()
        self.r.execute_request_basic(parser=parse)
        parse()

        self.r.extract_xml_entries(self.PHOTO_MAPPING, converter=self.r.convert_photo)

        output = {'result': self.r.context['processed_data']}
        if len(self.r.context['processed_data']) == self.PHOTOS_PER_REQUEST:
            output['next_token'] = next_token + self.PHOTOS_PER_REQUEST

        return output

    def refresh_token(self, refresh_token):
        return self._refresh_token(refresh_token, self._detect_refresh_token_error)

    def _detect_refresh_token_error(self, response):
        try:
            oauth2.refresh_token.detect_error(response)
        except oauth2.refresh_token.UnexpectedException:
            if response['error'] == 'internal_failure':
                raise ProviderTemporaryUnavailableProxylibError()
            else:
                raise

    def exchange_token(self):
        return self._authorization_code_to_token(self.r.access_token['value'])

    def get_token_info(self, need_client_id=True):
        self.r.compose_request(
            url_name='api_url_oauth2_v3',
            method='tokeninfo',
            add_access_token=False,
            additional_args={
                'access_token':  self.r.access_token['value'],
            },
        )

        def parse():
            if self.r.context['raw_response'].status == 400:
                raise InvalidTokenProxylibError()
            self.r.deserialize_json()
            self.r.parse_error_response()
        self.r.execute_request_basic(parser=parse)
        parse()

        self.r.extract_response_data(
            self.TOKEN_MAPPING,
            one=True,
            listed=False,
        )

        token_info = self.r.context['processed_data']
        token_info['scopes'] = token_info['scopes'].split()

        expires = token_info.get('expires')
        if expires is not None:
            token_info['expires'] = int(expires)

        client_id = token_info['client_id']
        if client_id != self.r.app.id:
            raise BadParametersProxylibError(
                'Token does not belong to the application',
            )
        return token_info


mapper.add_mapping(GoogleProxy.code, GoogleProxy)
