# -*- coding: utf-8 -*-
from __future__ import unicode_literals

from . import mapper
from .proxy import SocialProxy


UNKNOWN_AVATAR_SIZE = '0x0'


class LastFmProxy(SocialProxy):
    code = 'lf'

    SETTINGS = {
        'birthday_regexp': r'^(?P<year>\d{4})?\-?(?P<month>\d{2})\-(?P<day>\d{2})$',
        'gender_map': {'m': 'm', 'f': 'f'},

        'api_url': 'https://ws.audioscrobbler.com/2.0/',
        'should_sign': True,
        'sign_with_token': True,
        'signature_type': 'md5',
        'signature_key_value_separator': '',
        'signature_name': 'api_sig',

        'add_app_id': True,
        'app_id_key_name': 'api_key',

        'access_token_parameter_name': 'sk',

        # Комментарии к кодам ошибок: https://www.last.fm/api/errorcodes
        'error_codes_permission': [4],
        'error_codes_invalid_token': [6, 9, 10, 15, 26],
        'error_rate_limit_exceeded': [29],
    }

    PROFILE_MAPPING = {
        'name': 'userid',
        'realname': 'name',
        'gender': 'gender',
        'image': 'avatar',
    }

    AVATAR_MAPPING = {
        'small': '34x34',
        'medium': '64x64',
        'large': '174x174',
        'extralarge': '300x300',
    }

    def get_profile(self):
        """
        {
          "user": {
            "name": "Flidster",
            "realname": "Антон",
            "image": [
              {
                "#text": "",
                "size": "small"
              },
              {
                "#text": "",
                "size": "medium"
              },
              {
                "#text": "",
                "size": "large"
              },
              {
                "#text": "",
                "size": "extralarge"
              }
            ],
            "url": "http://www.last.fm/user/Flidster",
            "id": "56288542",
            "lang": "en",
            "country": "",
            "age": "24",
            "gender": "m",
            "subscriber": "0",
            "playcount": "512",
            "playlists": "1",
            "bootstrap": "0",
            "registered": {
              "#text": "2013-10-24 11:05",
              "unixtime": "1382612705"
            },
            "type": "user"
          }
        }
        """
        self.r.compose_request(url_name='api_url', method='user.getInfo', additional_args={'format': 'json'})

        self.r.execute_request_basic()
        self.r.deserialize_json()
        self.r.parse_error_response()
        self.r.extract_response_data(self.PROFILE_MAPPING, listed=False, one=True, converter=self.r.convert_profile,
                                     extract_field='user')
        self.r.convert_gender()
        self.convert_avatar()
        return self.r.context['processed_data']

    def convert_avatar(self):
        processed_avatars = dict()
        unprocessed_avatars = self.r.context['processed_data'].pop('avatar', [])
        for avatar in unprocessed_avatars:
            size_lastfm_units = avatar['size']
            size_yandex_units = self.AVATAR_MAPPING.get(size_lastfm_units, UNKNOWN_AVATAR_SIZE)
            avatar_url = avatar['#text']
            if avatar_url:
                processed_avatars[size_yandex_units] = avatar_url
        if processed_avatars:
            self.r.context['processed_data']['avatar'] = processed_avatars


mapper.add_mapping(LastFmProxy.code, LastFmProxy)
