# -*- coding: utf-8 -*-

from __future__ import unicode_literals

import json
import logging
import uuid

from passport.backend.social.common import oauth2
from passport.backend.social.common.exception import InvalidTokenProxylibError
from passport.backend.social.common.providers.MailRu import MailRu

from . import mapper
from .proxy import (
    save_method_info,
    SocialProxy,
)


logger = logging.getLogger(__name__)


_O1_HOSTNAME = 'www.appsmail.ru'


class MailRuProxy(SocialProxy):
    code = MailRu.code

    PRIVACY_LOCAL_BY_GLOBAL = {
        'friends': '4',
        'public': '2',
        'private': '0',
    }

    PRIVACY_GLOBAL_BY_LOCAL = dict((v, k) for k, v in PRIVACY_LOCAL_BY_GLOBAL.items())

    SETTINGS = {
        'birthday_regexp': r'^(?P<day>\d{2})\.(?P<month>\d{2})\.?(?P<year>\d{4})?$',
        'gender_map': {0: 'm', 1: 'f'},

        'api_url': 'http://%s/platform/api' % _O1_HOSTNAME,
        'oauth_refresh_token_url': 'https://%s/oauth/token' % _O1_HOSTNAME,

        'access_token_parameter_name': 'session_key',
        'should_sign': True,
        'signature_type': 'md5',
        'sign_with_token': True,
        'add_app_id': True,

        'error_codes_permission': [200],
        'error_codes_invalid_token': [102, 105],
        'error_user_blocked': [701],
        'album_not_exists': [100],
    }

    PROFILE_MAPPING = {
        'uid': 'userid',
        'first_name': 'firstname',
        'last_name': 'lastname',
        'nick': 'nickname',
        'sex': 'gender',
        'birthday': 'birthday',
        'email': 'email',
    }

    AVATAR_MAPPING = {
        'pic': 'avatar.90x0',
        'pic_small': 'avatar.45x0',
        'pic_big': 'avatar.600x0',
    }
    PROFILE_MAPPING.update(AVATAR_MAPPING)

    PHOTO_ALBUM_MAPPING = {
        'aid': 'aid',
        'title': 'title',
        'description': 'description',
        'created': 'created',
        'size': 'photo_count',
        'privacy': 'visibility',  # http://api.mail.ru/docs/reference/rest/photos.createAlbum/#params

        'link': 'url',
    }

    PHOTO_MAPPING = {
        'pid': 'pid',
        'title': 'caption',
        'created': 'created',

        'width': 'width',
        'height': 'height',

        'src_small': 'src_small',
        'src': 'src',
        'src_big': 'src_big',
    }

    PHOTOS_PER_REQUEST = 1000
    PHOTO_REQUESTS_PER_SESSION = 1

    def get_profile(self):
        self.r.compose_request(url_name='api_url', method='users.getInfo', additional_args={'secure': 1})
        self.r.execute_request_basic()
        self.r.deserialize_json()
        self.r.parse_error_response()
        self.r.extract_response_data(self.PROFILE_MAPPING, one=True, converter=self.r.convert_profile)
        self.r.convert_birthday()
        self.r.convert_gender()
        return self.r.context['processed_data']

    def get_mails_unread_count(self):
        self.r.compose_request(url_name='api_url', method='mail.getUnreadCount', additional_args={'secure': '1'})
        self.r.execute_request_basic()
        self.r.deserialize_json()
        self.r.parse_error_response()
        self.r.extract_mails_unread_count()
        return self.r.context['processed_data']

    def get_friends(self):
        self.r.compose_request(url_name='api_url', method='friends.get', additional_args={'secure': 1,
                                                                                          'ext': 1})
        self.r.execute_request_basic()
        self.r.deserialize_json()
        self.r.parse_error_response()
        self.r.extract_response_data(self.PROFILE_MAPPING, one=False)
        self.r.convert_friends_gender()
        self.r.convert_friends_birthday()
        return self.r.context['processed_data']

    def get_photo_albums(self, userid):
        self.r.compose_request(url_name='api_url', method='photos.getAlbums',
                               additional_args={'secure': 0}, userid=userid)
        self.r.execute_request_basic()
        self.r.deserialize_json()
        self.r.parse_error_response()
        self.r.extract_response_data(self.PHOTO_ALBUM_MAPPING, one=False)
        self.r.convert_photo_albums_privacy(self.PRIVACY_GLOBAL_BY_LOCAL)

        # не отображаем спец альбомы в общем списке
        self.r.context['processed_data'] = [x for x in self.r.context['processed_data']
                                            if x['aid'] not in ['_myface', '_myphoto']]

        return self.r.context['processed_data']

    def create_photo_album(self, title, userid, description=None, privacy=None):
        additional_args = {
            'secure': 0,
            'title': title,
            'aid': uuid.uuid4().hex,  # случайная строка
        }
        if description:
            additional_args['description'] = description
        self.update_request_privacy(additional_args, privacy, 'privacy')
        self.r.compose_request(url_name='api_url', method='photos.createAlbum',
                               additional_args=additional_args, userid=userid)
        self.r.execute_request_basic()
        self.r.deserialize_json()
        self.r.parse_error_response()
        return {'aid': self.r.context['data']['aid']}

    @save_method_info
    def get_photos(self, aid, userid, next_token=None, debug=False):

        photos_per_request = self.PHOTOS_PER_REQUEST if not debug else 2

        if aid == 'personal':
            aid = '_myphoto'  # Альбом "Фотографии со мной"

        try:
            offset = int(str(next_token))
        except ValueError:
            offset = 0

        request_args = {
            'secure': 0,
            'aid': aid,
            'limit': photos_per_request,
            'offset': offset,
        }

        self.r.compose_request(
            url_name='api_url',
            method='photos.get',
            additional_args=request_args,
            userid=userid,
        )
        self.r.execute_request_basic()
        self.r.deserialize_json()
        self.r.parse_error_response()
        self.r.extract_response_data(self.PHOTO_MAPPING, one=False, converter=self.r.convert_photo)

        output = {'result': self.r.context['processed_data']}
        if len(self.r.context['processed_data']) == photos_per_request:
            output['next_token'] = offset + photos_per_request

        return output

    @save_method_info
    def get_user_photos(self, userid, next_token=None):
        return self.get_photos('_myface', userid, next_token=next_token)

    @save_method_info
    def photo_post_get_request(self, aid, caption=None):
        additional_args = {'secure': 1, 'aid': aid}
        if caption:
            additional_args['name'] = caption

        self.r.compose_request(url_name='api_url', method='photos.upload',
                               additional_args=additional_args)

        output = {
            'url': self.r.context['request']['url'],
            'image_name': 'img_file',
        }

        return output

    def photo_post_commit(self, upload_response):
        self.r.context['data'] = json.loads(upload_response)
        self.r.parse_error_response()
        output = {'pid': self.r.context['data']['pid']}
        return output

    def refresh_token(self, refresh_token):
        try:
            return self._refresh_token(refresh_token)
        except oauth2.refresh_token.InvalidRequest as e:
            raise InvalidTokenProxylibError(e.error_description)


class MailRuO2Proxy(SocialProxy):
    code = MailRu.code

    SETTINGS = {
        'oauth_refresh_token_url': 'https://o2.mail.ru/token',

        'error_codes_invalid_token': [6],
        'error_codes_service_unavailable': [5],
    }

    PROFILE_MAPPING = {
        'id': 'userid',
        'email': 'email',
        'first_name': 'firstname',
        'last_name': 'lastname',
        'gender': 'gender',
    }

    def get_profile(self):
        self.r.compose_request(base_url='https://o2.mail.ru/userinfo')

        def parse():
            self.r.deserialize_json()
            self._detect_error_in_response(self.r.context['data'])
        self.r.execute_request_basic(parser=parse)
        parse()

        self.r.extract_response_data(self.PROFILE_MAPPING, one=True, listed=False)

        profile = self.r.context['processed_data'].copy()
        profile['username'] = profile['email']

        return profile

    def refresh_token(self, refresh_token):
        return self._refresh_token(
            refresh_token,
            # Mail.ru не соблюдают OAuth, поэтому приходится использовать
            # собственный обработчик ошибок.
            self._detect_error_in_response,
        )

    def _detect_error_in_response(self, response):
        if 'error_code' in response:
            error_message = response.get('error_description', '')
            self.r.raise_correct_exception(response['error_code'], error_message)


custom_mapping = {'o2': MailRuO2Proxy}
mapper.add_mapping(MailRu.code, MailRuProxy, mapping=custom_mapping)
