# -*- coding: utf-8 -*-
from __future__ import unicode_literals

from passport.backend.social.common import oauth2
from passport.backend.social.common.exception import ProviderTemporaryUnavailableProxylibError

from . import mapper
from .proxy import SocialProxy


class MicrosoftProxy(SocialProxy):
    code = 'ms'

    SETTINGS = {
        'gender_map': {'male': 'm', 'female': 'f'},  # пальцем в небо: в доке поле не описано, на практике отдаётся null
        'api_url': 'https://apis.live.net/v5.0/%(method)s',
        'avatar_url': 'https://apis.live.net/v5.0/%(userid)s/picture?type=%(size)s',
        'oauth_refresh_token_url': 'https://login.live.com/oauth20_token.srf',

        'error_codes_invalid_token': ['request_token_invalid'],
        'error_codes_service_unavailable': ['server_internal_error'],
    }

    PROFILE_MAPPING = {
        'id': 'userid',
        'first_name': 'firstname',
        'last_name': 'lastname',
        'gender': 'gender',
        'birth_day': 'birth_day',
        'birth_month': 'birth_month',
        'birth_year': 'birth_year',
        'emails': 'emails',
        'locale': 'locale',
        'updated_time': 'updated',
    }

    def get_profile(self):
        self.r.compose_request(url_name='api_url', method='me')
        self.r.execute_request_basic()
        self.r.deserialize_json()
        self.r.parse_error_response()

        self.r.extract_response_data(self.PROFILE_MAPPING, one=True, listed=False)
        self.r.convert_birthday()
        self.r.convert_gender()
        self.r.convert_avatar(avatar_url=self.SETTINGS['avatar_url'])
        self.r.convert_emails()

        return self.r.context['processed_data']

    def refresh_token(self, refresh_token):
        return self._refresh_token(refresh_token, self._detect_refresh_token_error)

    def _detect_refresh_token_error(self, response):
        try:
            oauth2.refresh_token.detect_error(response)
        except oauth2.refresh_token.UnexpectedException:
            if response['error'] == 'server_error':
                raise ProviderTemporaryUnavailableProxylibError()
            else:
                raise


mapper.add_mapping(MicrosoftProxy.code, MicrosoftProxy)
