# -*- coding: utf-8 -*-

from __future__ import unicode_literals

from passport.backend.social.common import validators
from passport.backend.social.common.exception import BadParametersProxylibError
from passport.backend.social.common.misc import expires_in_to_expires_at
from passport.backend.social.common.providers.Mts import Mts
from passport.backend.social.proxylib import mapper
from passport.backend.social.proxylib.proxy import (
    SocialPackedTokenParser,
    SocialPackedTokenVersionForm,
    SocialProxy,
)
from passport.backend.utils.common import noneless_dict


def parse_mts_client_token(doc):
    token_dict = SocialPackedTokenParser({
        1: MtsClientTokenV1Form,
    }).parse(doc)
    return MtsClientToken.from_dict(token_dict)


class MtsProxy(SocialProxy):
    code = Mts.code

    SETTINGS = {
        'api_url': 'https://login.mts.ru/oauth2/api',
        'birthday_regexp': r'^(?P<year>\d{4})\-(?P<month>\d{2})\-(?P<day>\d{2})T00:00:00$',
        'oauth_auth_over_header': True,
        'error_codes_invalid_token': {401},
        'oauth_refresh_token_url': 'https://login.mts.ru/amserver/oauth2/token',
    }

    PROFILE_MAPPING = {
        'profile:name:type': 'type',
        'profile:name': 'person_name',
        'profile:name:org': 'organization_name',
        # У организаций нет дня рождения
        'profile:birthday': 'birthday',
        'mobile:phone': 'phone',
        'mobile:account': 'personal_id',
    }

    TOKEN_MAPPING = {
        'scope': 'scopes',
        'exp': 'expires',
        'aud': 'client_id',
    }

    def get_profile(self):
        self.r.compose_request(url_name='api_url')
        self.r.execute_request_basic()
        self.r.parse_error_response()
        self.r.deserialize_json()
        self.r.extract_response_data(self.PROFILE_MAPPING, converter=self.r.convert_profile, listed=False, one=True)
        self.r.convert_birthday()
        return self.r.context['processed_data']

    def get_token_info(self, need_client_id=True):
        self.r.compose_request(url_name='api_url')
        self.r.execute_request_basic()
        self.r.parse_error_response()
        self.r.deserialize_json()
        self.r.extract_response_data(
            self.TOKEN_MAPPING,
            one=True,
            listed=False,
        )

        token_info = self.r.context['processed_data']

        expires = token_info.get('expires')
        if expires is not None:
            token_info['expires'] = int(expires)

        client_id = token_info['client_id']
        if client_id != self.r.app.id:
            raise BadParametersProxylibError(
                'Token does not belong to the application',
            )
        return token_info

    def refresh_token(self, refresh_token):
        return self._refresh_token(refresh_token)


class MtsV3Proxy(SocialProxy):
    code = Mts.code

    SETTINGS = {
        'api_url': 'https://login.mts.ru/amserver/oauth2/realms/root/realms/users/userinfo',
        'birthday_regexp': r'^(?P<year>\d{4})-(?P<month>\d{2})-(?P<day>\d{2})$',
        'oauth_auth_over_header': True,
        'error_codes_invalid_token': {401},
        'oauth_refresh_token_url': 'https://login.mts.ru/amserver/oauth2/realms/root/realms/users/access_token',
    }

    PROFILE_MAPPING = {
        'account:number': 'personal_id',
        'birthday': 'birthday',
        'given_name': 'firstname',
        'is_organization': 'is_organization',
        'last_name': 'lastname',
        'organization': 'organization_name',
        'phone': 'phone',
    }

    def get_profile(self):
        self.r.compose_request(url_name='api_url')
        self.r.execute_request_basic()
        self.r.parse_error_response()
        self.r.deserialize_json()
        self.r.extract_response_data(self.PROFILE_MAPPING, converter=self.r.convert_profile_v3, listed=False, one=True)
        return self.r.context['processed_data']

    def refresh_token(self, refresh_token):
        return self._refresh_token(refresh_token)

    def exchange_token(self):
        client_token = parse_mts_client_token(self.r.access_token['value'])
        return noneless_dict(
            expires=expires_in_to_expires_at(client_token.expires_in),
            refresh=client_token.refresh_token,
            scope=client_token.scope,
            value=client_token.access_token,
        )


class MtsClientToken(object):
    def __init__(self):
        self.access_token = None
        self.expires_in = None
        self.refresh_token = None
        self.scope = None
        self.version = None

    @classmethod
    def from_dict(cls, d):
        self = MtsClientToken()
        self.access_token = d['access_token']
        self.expires_in = d['expires_in']
        self.refresh_token = d['refresh_token']
        self.scope = d['scope']
        self.version = d['version']
        return self


class MtsClientTokenV1Form(SocialPackedTokenVersionForm):
    access_token = validators.Token()
    expires_in = validators.Int(if_empty=None, if_missing=None)
    refresh_token = validators.Token(if_empty=None, if_missing=None)
    scope = validators.ScopeString(if_empty=None, if_missing=None)


custom_mapping = {'mts_v3': MtsV3Proxy}
mapper.add_mapping(MtsProxy.code, MtsProxy, mapping=custom_mapping)
