# -*- coding: utf-8 -*-
from __future__ import unicode_literals

import json
import re

from passport.backend.social.common.exception import ProviderCommunicationProxylibError

from . import mapper
from .proxy import (
    paginated,
    save_method_info,
    SocialProxy,
)


class OdnoklassnikiProxy(SocialProxy):
    code = 'ok'

    PRIVACY_LOCAL_BY_GLOBAL = {
        'friends': 'friends',
        'public': 'public',
        'private': 'friends',
    }

    PRIVACY_GLOBAL_BY_LOCAL = {
        'friends': 'friends',
        'colleague': 'friends',
        'classmate': 'friends',
        'cursemate': 'friends',
        'public': 'public',
    }

    SETTINGS = {
        'birthday_regexp': r'^(?P<year>\d{4})?\-?(?P<month>\d{2})\-(?P<day>\d{2})$',
        'gender_map': {'male': 'm', 'female': 'f'},
        'avatars_exclude_regexp': re.compile(r'^.*/stub_\d+x\d+\.gif$'),

        'api_url': 'http://api.odnoklassniki.ru/fb.do',
        'should_sign': True,
        'add_application_key': True,

        'error_codes_permission': [10],
        'error_codes_invalid_token': [102, 103],
        'album_not_exists': [120],
    }

    PROFILE_MAPPING = {
        'uid': 'userid',
        'first_name': 'firstname',
        'last_name': 'lastname',
        'gender': 'gender',
        'birthday': 'birthday',
        'name': 'nickname',
        'locale': 'locale',
        'location': 'location',
    }

    GROUP_MAPPING = {
        'uid': 'gid',
        'name': 'name',
        'private': 'private',
    }

    AVATAR_MAPPING = {
        'pic_1': 'avatar.50x50',
        'pic_2': 'avatar.50x0',
        'pic_3': 'avatar.190x190',
        'pic_4': 'avatar.640x0',
    }

    PROFILE_MAPPING.update(AVATAR_MAPPING)

    PHOTOS_PER_REQUEST = 100
    PHOTO_REQUESTS_PER_SESSION = 15

    PHOTO_ALBUM_MAPPING = {
        'aid': 'aid',
        'title': 'title',
        'description': 'description',
        'created': 'created',
        'photos_count': 'photo_count',
        'type': 'type',  # public, friends
        'types': 'types',
    }

    PHOTO_MAPPING = {
        'id': 'pid',
        'text': 'caption',

        'standard_width': 'width',
        'standard_height': 'height',

        'pic50x50': 'pic50x50',
        'pic128x128': 'pic128x128',
        # 'pic180min': 'pic180min',
        'pic190x190': 'pic190x190',
        'pic640x480': 'pic640x480',
        'pic1024x768': 'pic1024x768',
    }

    def get_profile(self, userid=None):

        if userid is None:
            self.r.compose_request(url_name='api_url', method='users.getCurrentUser', additional_args={'format': 'json'})
        else:
            # more informative way (lets us to query user location, locale...)
            self.r.compose_request(url_name='api_url', method='users.getInfo', fields=self.PROFILE_MAPPING.keys(),
                                   additional_args={'uids': str(userid), 'format': 'json', 'emptyPictures': 'true'})
        self.r.execute_request_basic()
        self.r.deserialize_json()
        self.r.parse_error_response()
        self.r.extract_response_data(self.PROFILE_MAPPING, listed=userid is not None,
                                     one=True, converter=self.r.convert_profile)
        self.r.filter_stub_avatars()
        self.r.convert_birthday()
        self.r.convert_gender()
        return self.r.context['processed_data']

    def get_groups(self):
        self.r.compose_request(url_name='api_url', method='group.getUserGroupsV2', additional_args={'format': 'json'})
        self.r.execute_request_basic()
        self.r.deserialize_json()
        self.r.parse_error_response()

        gids = [g['groupId'] for g in self.r.context['data']['groups']]

        self.r.compose_request(url_name='api_url', method='group.getInfo',
                               additional_args={'format': 'json', 'uids': ','.join(gids),
                                                'fields': 'uid,name,private'})
        self.r.execute_request_basic()
        self.r.deserialize_json()
        self.r.parse_error_response()
        self.r.extract_response_data(self.GROUP_MAPPING, converter=self.r.convert_group)
        return self.r.context['processed_data']

    @paginated
    @save_method_info
    def get_friends(self):
        self.r.compose_request(url_name='api_url', method='friends.get', additional_args={'format': 'json'})
        self.r.execute_request_basic()
        self.r.deserialize_json()
        self.r.parse_error_response()

        userids = self.r.context['data']

        processed = 0
        # максимальное количество запрашиваемых userid за раз - 100
        ITEMS_PER_REQUIEST = 100
        while processed < len(userids):

            self.r.compose_request(url_name='api_url', method='users.getInfo', fields=self.PROFILE_MAPPING.keys(),
                                   additional_args={'uids': ','.join(userids[processed:processed+ITEMS_PER_REQUIEST]),
                                                    'format': 'json'})
            self.r.execute_request_basic()
            self.r.deserialize_json()
            self.r.parse_error_response()
            self.r.extract_response_data(self.PROFILE_MAPPING, one=False, converter=self.r.convert_profile)

            processed += ITEMS_PER_REQUIEST
            self.r.convert_friends_birthday()
            self.r.convert_friends_gender()

            yield self.r.context['processed_data']

    def get_marks_unread_count(self):
        self.r.compose_request(url_name='api_url', method='events.get', additional_args={'format': 'json'})
        self.r.execute_request_basic()
        self.r.deserialize_json()
        self.r.parse_error_response()
        self.r.extract_events_count('marks')
        return self.r.context['processed_data']

    def get_notifications_unread_count(self):
        self.r.compose_request(url_name='api_url', method='events.get', additional_args={'format': 'json'})
        self.r.execute_request_basic()
        self.r.deserialize_json()
        self.r.parse_error_response()
        self.r.extract_events_count('notifications')
        return self.r.context['processed_data']

    def get_messages_unread_count(self):
        self.r.compose_request(url_name='api_url', method='events.get', additional_args={'format': 'json'})
        self.r.execute_request_basic()
        self.r.deserialize_json()
        self.r.parse_error_response()
        self.r.extract_events_count('messages')
        return self.r.context['processed_data']

    @paginated
    @save_method_info
    def get_photo_albums(self):
        ITEMS_PER_REQUEST = 100
        pagingAnchor = ''
        while True:
            self.r.compose_request(url_name='api_url', method='photos.getAlbums',
                                   fields=['album.' + x for x in self.PHOTO_ALBUM_MAPPING.keys()],
                                   additional_args={'format': 'json', 'count': ITEMS_PER_REQUEST,
                                                    'pagingAnchor': pagingAnchor})
            self.r.execute_request_basic()
            self.r.deserialize_json()
            self.r.parse_error_response()
            self.r.extract_response_data(self.PHOTO_ALBUM_MAPPING, one=False, converter=self.r.convert_album,
                                         extract_field='albums')
            self.r.convert_photo_albums_privacy(self.PRIVACY_GLOBAL_BY_LOCAL)

            yield self.r.context['processed_data']

            if not self.r.context['data'].get('hasMore'):
                break
            pagingAnchor = self.r.context['data']['pagingAnchor']

    @save_method_info
    def create_photo_album(self, title, description=None, privacy=None):
        additional_args = {
            'title': title,

            # обязательный параметр, поэтому ставим значение по умолчанию
            'type': 'friends',
        }
        if description:
            additional_args['description'] = description
        self.update_request_privacy(additional_args, privacy, 'type')
        self.r.compose_request(url_name='api_url', method='photos.createAlbum', additional_args=additional_args)
        self.r.execute_request_basic()
        self.r.deserialize_json()
        self.r.parse_error_response()
        return {'aid': str(self.r.context['data'])}

    @save_method_info
    def get_photos(self, aid, next_token=None, debug=False):

        output = {'result': []}

        # получение личных фотографий пользователя
        if aid == 'personal':
            aid = ''

        args = {
            'format': 'json',
            'count': self.PHOTOS_PER_REQUEST if not debug else 1,
            'anchor': next_token if next_token else '',
            'aid': aid,
        }
        new_next_token = None

        for _ in range(self.PHOTO_REQUESTS_PER_SESSION if not debug else 2):
            self.r.compose_request(url_name='api_url', method='photos.getPhotos',
                                   fields=['photo.' + x for x in self.PHOTO_MAPPING.keys()],
                                   additional_args=args)
            self.r.execute_request_basic()
            self.r.deserialize_json()
            self.r.parse_error_response()
            self.r.extract_response_data(self.PHOTO_MAPPING, one=False, converter=self.r.convert_photo,
                                         extract_field='photos')

            output['result'].extend(self.r.context['processed_data'])

            if not self.r.context['data'].get('hasMore'):
                break
            args['anchor'] = self.r.context['data']['anchor']
        else:
            new_next_token = args['anchor']

        output['next_token'] = new_next_token

        return output

    @save_method_info
    def get_user_photos(self):
        return self.get_photos('tags')  # pragma: no cover

    @save_method_info
    def photo_post_get_request(self, aid):
        self.r.compose_request(url_name='api_url', method='photosV2.getUploadUrl',
                               additional_args={'format': 'json', 'aid': aid})

        self.r.execute_request_basic()
        self.r.deserialize_json()
        self.r.parse_error_response()

        output = {
            'url': self.r.context['data']['upload_url'],
            'image_name': 'pic1',
        }
        return output

    @save_method_info
    def photo_post_commit(self, upload_response, caption=None):
        self.r.context['data'] = json.loads(upload_response)
        self.r.parse_error_response()
        key = self.r.context['data']['photos'].keys()[0]
        token = self.r.context['data']['photos'][key]['token']

        additional_args = {
            'photo_id': key,
            'token': token,
            'format': 'json',
        }
        if caption:
            additional_args['comment'] = caption

        self.r.compose_request(url_name='api_url', method='photosV2.commit',
                               additional_args=additional_args)

        self.r.execute_request_basic()
        self.r.deserialize_json()
        self.r.parse_error_response()
        photo = self.r.context['data']['photos'][0]
        if photo['status'] == 'SUCCESS':
            return {'pid': photo['assigned_photo_id']}
        raise ProviderCommunicationProxylibError('Failed to upload photo, reason unknown.')

    def wall_post(self, text=None, picture=None, name=None, caption=None, description=None, link=None):
        media = []

        if link:
            media.append({
                'type': 'link',
                'url': link,
            })

        if text:
            media.append({
                'type': 'text',
                'text': text,
            })

        args = {
            'format': 'json',
            'type': 'USER',
            'attachment': json.dumps({'media': media}),
        }

        self.r.compose_request(url_name='api_url', method='mediatopic.post', additional_args=args)

        self.r.execute_request_basic()
        self.r.deserialize_json()
        self.r.parse_error_response()
        return {'post_id': str(self.r.context['data'])}


mapper.add_mapping(OdnoklassnikiProxy.code, OdnoklassnikiProxy)
