# -*- coding: utf-8 -*-

from __future__ import unicode_literals

import re

from passport.backend.social.common.useragent import Request
from passport.backend.social.proxylib import mapper
from passport.backend.social.proxylib.proxy import (
    paginated,
    SocialProxy,
)


class TwitterProxy(SocialProxy):
    code = 'tw'
    SETTINGS = {
        'avatars_exclude_regexp': re.compile(r'^.*/default_profile_images/default_profile_[^/]+\.png$'),

        'error_codes_permission': [179],
        'error_codes_invalid_token': [89, 215],
        'error_rate_limit_exceeded': [88],
        'error_codes_service_unavailable': [131],
        'api_url': 'https://api.twitter.com/1.1/%(method)s',
    }

    PROFILE_MAPPING = {
        'id': 'userid',
        'name': 'name',
        'screen_name': 'username',
        'followers_count': 'followers_count',
        'statuses_count': 'messages_count',
        'friends_count': 'friends_count',
    }

    AVATAR_MAPPING = {
        'profile_image_url': 'avatar.48x48',
    }
    PROFILE_MAPPING.update(AVATAR_MAPPING)

    def get_profile(self):
        self.r.compose_profile_url_oauth1('api_url', 'account/verify_credentials.json')
        self.r.execute_request_basic()
        self.r.deserialize_json()
        self.r.parse_error_response()
        self.r.extract_response_data(self.PROFILE_MAPPING, converter=self.r.convert_profile, listed=False, one=True)
        self.r.filter_stub_avatars()
        return self.r.context['processed_data']

    @paginated
    def get_friends(self):
        args = {'count': 200, 'skip_status': 'true', 'include_user_entities': 'false', 'cursor': -1}
        while True:
            if args['cursor'] == 0:
                break
            self.r.compose_profile_url_oauth1('api_url', 'friends/list.json', additional_args=args)
            self.r.execute_request_basic()
            self.r.deserialize_json()
            self.r.parse_error_response()
            args['cursor'] = self.r.context['data']['next_cursor']
            self.r.extract_response_data(self.PROFILE_MAPPING, converter=self.r.convert_profile, extract_field='users')

            yield self.r.context['processed_data']

    def get_auth_headers(self, method, url, post_data=None):
        signer = self.r.get_request_signer()
        unsigned_request = Request(method=method, url=url, data=post_data)
        request = signer.get_signed_request(unsigned_request)
        return request.headers

    def wall_post(self, text=None, picture=None, name=None, caption=None, description=None, link=None):
        not_empty_items = filter(None, [text, link])

        data = {
            'status': ' '.join(not_empty_items),
        }

        self.r.compose_profile_url_oauth1(
            'api_url',
            'statuses/update.json',
            http_method='POST',
            additional_post_data=data,
        )
        self.r.execute_request_basic()
        self.r.deserialize_json()
        self.r.parse_error_response()
        return {'post_id': str(self.r.context['data']['id'])}


mapper.add_mapping(TwitterProxy.code, TwitterProxy)
