# -*- coding: utf-8 -*-

from __future__ import unicode_literals

import re

from . import mapper
from .proxy import SocialProxy


class VkontakteProxy(SocialProxy):
    code = 'vk'

    PRIVACY_LOCAL_BY_GLOBAL = {
        'friends': 'friends',
        'public': 'all',
        'private': 'only_me',
    }

    PRIVACY_GLOBAL_BY_LOCAL = {
        'all': 'public',
        'friends': 'friends',
        'friends_of_friends': 'friends',
        'friends_of_friends_only': 'friends',
        'nobody': 'private',
        'only_me': 'private',
    }

    SETTINGS = {
        'birthday_regexp': r'^(?P<day>\d{1,2})\.(?P<month>\d{1,2})\.?(?P<year>\d{4})?$',
        'gender_map': {2: 'm', 1: 'f'},
        'api_url': 'https://api.vk.com/method/%(method)s',
        'avatars_exclude_regexp': re.compile(
            r'^https?://vk\.com/.*$|'
            r'^https?://www\.vk\.com/.*$',
        ),
        'application_key_key_name': 'access_token',

        'error_codes_permission': [7, 15, 30, 200],
        'error_codes_invalid_token': [5, 39],
        'error_rate_limit_exceeded': [6, 29],
        'error_captcha_needed': [14],
        'error_validation_required': [17],
        'error_codes_service_unavailable': [1, 10],
        'error_user_blocked': [18, 3610],
    }

    PROFILE_MAPPING = {
        'id': 'userid',
        'first_name': 'firstname',
        'last_name': 'lastname',
        'sex': 'gender',
        'bdate': 'birthday',

        # Это username
        'domain': 'domain',

        # Это отчество или прозвище
        'nickname': 'nickname',

        # Для того чтобы в ответе вернулись mobile_phone и home_phone у ВК
        # следует запросить поле contacts. А чтобы поля mobile_phone и
        # home_phone правильно распрасились их также нужно указать здесь, хоть
        # у ВК и нельзя запрашивать такие поля.
        'contacts': 'contacts',
        'mobile_phone': 'mobile_phone',
        'home_phone': 'home_phone',

        'country': 'country',
        'city': 'city',
        'universities': 'universities',
    }

    AVATAR_MAPPING = {
        'photo_50': 'avatar.50x50',
        'photo_100': 'avatar.100x100',
        'photo_200': 'avatar.200x200',
        'photo_200_orig': 'avatar.200x0',
        'photo_400_orig': 'avatar.400x0',
        'photo_max_orig': 'avatar.0x0',
    }

    PROFILE_MAPPING.update(AVATAR_MAPPING)

    TOKEN_MAPPING = {'id': 'client_id'}

    TOKEN_SCOPE_IDS = {
        1: 'notify',
        2: 'friends',
        4: 'photos',
        8: 'audio',
        16: 'video',
        32: 'offers',
        64: 'questions',
        128: 'pages',
        1024: 'status',
        2048: 'notes',
        4096: 'messages',
        8192: 'wall',
        32768: 'ads',
        65536: 'offline',
        131072: 'docs',
        262144: 'groups',
        524288: 'notifications',
        1048576: 'stats',
        4194304: 'email',
        134217728: 'market',
    }

    def get_profile(self):
        self.r.compose_request(
            url_name='api_url',
            method='users.get',
            fields=self.PROFILE_MAPPING.keys(),
            additional_args={'v': '5.131'},
        )
        self.r.execute_request_basic()
        self.r.deserialize_json()
        self.r.parse_error_response()
        self.r.extract_response_data(
            self.PROFILE_MAPPING,
            one=True,
            converter=self.r.convert_profile,
            extract_field='response',
        )
        self.r.filter_stub_avatars()
        self.r.convert_birthday()
        self.r.convert_gender()
        self.r.extract_location()
        return self.r.context['processed_data']

    def get_friends(self, userid=None, public_only=False):
        additional_args = {'v': '5.131'}
        if public_only:
            # https://st.yandex-team.ru/PASSP-12076
            add_access_token = False
            if userid is None:
                # userid неизвестен, когда пользователь пришёл по токену.
                response = self.get_profile()
                userid = response['userid']
            additional_args['user_id'] = userid
            add_application_key = True
        else:
            add_access_token = True
            add_application_key = False

        self.r.compose_request(
            url_name='api_url',
            method='friends.get',
            fields=self.PROFILE_MAPPING.keys(),
            additional_args=additional_args,
            add_access_token=add_access_token,
            add_application_key=add_application_key,
        )

        self.r.execute_request_basic()
        self.r.deserialize_json()
        self.r.parse_error_response()
        self.r.extract_response_data(
            self.PROFILE_MAPPING,
            one=False,
            converter=self.r.convert_profile,
            extract_field=['response', 'items'],
        )
        self.r.convert_friends_birthday()
        self.r.convert_friends_gender()
        self.r.extract_friends_location()
        return self.r.context['processed_data']

    def get_token_info(self, need_client_id=True):
        token_info = {}

        if need_client_id:
            self.r.compose_request(
                url_name='api_url',
                method='apps.get',
                additional_args={'v': '5.131'},
            )
            self.r.execute_request_basic()
            self.r.deserialize_json()
            self.r.parse_error_response()
            self.r.extract_response_data(self.TOKEN_MAPPING, one=True, listed=True, extract_field=['response', 'items'])

            token_info['client_id'] = str(self.r.context['processed_data']['client_id'])

        self.r.compose_request(
            url_name='api_url',
            method='account.getAppPermissions',
            additional_args={'v': '5.131'},
        )
        self.r.execute_request_basic()
        self.r.deserialize_json()
        self.r.parse_error_response()
        self.r.context['processed_data'] = {}
        self.r.get_token_scopes(self.TOKEN_SCOPE_IDS)

        token_info['scopes'] = self.r.context['processed_data']['scopes']

        return token_info


mapper.add_mapping(VkontakteProxy.code, VkontakteProxy)
