# -*- coding: utf-8 -*-

from __future__ import unicode_literals

import re

from passport.backend.social.common import oauth2

from . import mapper
from .proxy import SocialProxy


class YahooProxy(SocialProxy):
    code = 'yh'

    SETTINGS = {
        'oauth_auth_over_header': True,
        'gender_map': {'M': 'm', 'F': 'f'},
        'oauth_refresh_token_url': 'https://api.login.yahoo.com/oauth2/get_token',
        'error_codes_invalid_token': [401],
        'avatars_exclude_regexp': re.compile('.*default_user_profile_pic_'),
    }

    PROFILE_MAPPING = {
        'sub': 'userid',
        'given_name': 'firstname',
        'family_name': 'lastname',
        'nickname': 'username',
        'locale': 'locale',
        'email': '_email',
        'email_verified': '_email_verified',
        'picture': 'avatar.0x0',
    }

    def get_profile(self):
        self.r.compose_request(
            base_url='https://api.login.yahoo.com/openid/v1/userinfo',
            additional_args={
                'format': 'json',
            },
        )
        self.r.execute_request_basic()
        self.r.parse_http_error_response()
        self.r.deserialize_json()
        self.r.extract_response_data(
            self.PROFILE_MAPPING,
            one=True,
            listed=False,
            converter=self._convert_profile,
        )

        return self.r.context['processed_data']

    def refresh_token(self, refresh_token):
        return self._refresh_token(refresh_token, self._detect_refresh_token_error)

    def _detect_refresh_token_error(self, response):
        try:
            oauth2.refresh_token.detect_error(response)
        except oauth2.refresh_token.UnexpectedException as e:
            if response['error'] == 'INVALID_REFRESH_TOKEN':
                raise oauth2.refresh_token.InvalidGrant.from_exception(e)
            elif response['error'] == 'ACCOUNT_NOT_AUTHORIZED':
                raise oauth2.refresh_token.InvalidGrant.from_exception(e)
            else:
                raise

    def _convert_profile(self, profile):
        email = profile.pop('_email', None)
        email_verified = profile.pop('_email_verified', None)
        if email_verified and email:
            profile['email'] = email
        self.r.filter_stub_avatars(profile)


mapper.add_mapping(YahooProxy.code, YahooProxy)
