# -*- coding: utf-8 -*-

from __future__ import unicode_literals

from urlparse import urljoin

from passport.backend.social.common.builders.blackbox import (
    Blackbox,
    BlackboxOauthTokenInvalidError,
    BlackboxTemporaryError,
    check_oauth_response,
)
from passport.backend.social.common.exception import (
    InvalidTokenProxylibError,
    ProviderTemporaryUnavailableProxylibError,
)
from passport.backend.social.common.providers.Yandex import Yandex
from passport.backend.social.common.social_config import social_config
from passport.backend.social.common.useragent import get_http_pool_manager

from . import mapper
from .proxy import SocialProxy


_AVATAR_SIZE = {
    'small': '25x25',
    'mini': '16x16',
    'islands-small': '28x28',
    'islands-34': '34x34',
    'islands-middle': '42x42',
    'islands-50': '50x50',
    'islands-retina-small': '56x56',
    'islands-68': '68x68',
    'islands-75': '75x75',
    'islands-retina-middle': '84x84',
    'islands-retina-50': '100x100',
    'islands-300': '300x300',
    'islands-200': '200x200',
    'islands-150': '150x150',
}


class YandexProxy(SocialProxy):
    code = Yandex.code

    @property
    def SETTINGS(self):
        return {
            'access_token_parameter_name': 'oauth_token',
            'gender_map': {'male': 'm', 'female': 'f'},
            'birthday_regexp': r'^(?P<year>\d{4})\-(?P<month>\d{2})\-(?P<day>\d{2})$',
            'get_profile_url': social_config.yandex_get_profile_url,
            'avatar_url_template': social_config.yandex_avatar_url_template,
            'error_codes_invalid_token': {401},
            'error_codes_service_unavailable': {503},
        }

    PROFILE_MAPPING = {
        'id': 'userid',
        'first_name': 'firstname',
        'last_name': 'lastname',
        'display_name': 'username',
        'sex': 'gender',
        'birthday': 'birthday',
        'default_email': 'email',

        'default_avatar_id': 'default_avatar_id',
        'is_avatar_empty': 'is_avatar_empty',
    }

    def __init__(self, *args, **kwargs):
        super(YandexProxy, self).__init__(*args, **kwargs)
        self.blackbox = Blackbox(get_http_pool_manager())

    def get_profile(self):
        self.r.compose_request(url_name='get_profile_url')

        self.r.execute_request_basic()
        self.r.parse_response()

        # В случае отказа отдаётся 400/500 и пустая строка, мы преобразуем в
        # ошибку о неправильном JSON-документе.
        self.r.extract_response_data(
            self.PROFILE_MAPPING,
            one=True,
            listed=False,
        )
        self.r.convert_gender()
        self.r.convert_birthday()

        self._build_avatars()

        return self.r.context['processed_data']

    def get_token_info(self, need_client_id=True):
        try:
            response = self.blackbox.oauth(
                oauth_token=self.r.access_token['value'],
                dbfields=[],
                attributes=[],
            )
        except BlackboxTemporaryError:
            raise ProviderTemporaryUnavailableProxylibError()

        try:
            check_oauth_response(response)
        except BlackboxOauthTokenInvalidError as e:
            raise InvalidTokenProxylibError(str(e))

        token_info = dict(
            client_id=response['oauth']['client_id'],
            scopes=response['oauth']['scope'],
        )
        return token_info

    def _build_avatars(self):
        avatar_id = self.r.context['processed_data'].get('default_avatar_id')
        is_avatar_empty = self.r.context['processed_data'].get('is_avatar_empty')
        if avatar_id and not is_avatar_empty:
            avatars = Avatars.from_avatar_id(avatar_id, self.SETTINGS['avatar_url_template'])
            self.r.context['processed_data']['avatar'] = avatars.as_dict()


class Avatars(object):
    def __init__(self):
        self._size_to_url = dict()

    @classmethod
    def from_avatar_id(cls, avatar_id, avatar_url_template=None):
        if avatar_url_template is None:
            avatar_url_template = social_config.yandex_avatar_url_template
        base_url = avatar_url_template % avatar_id

        avatars = Avatars()
        for name, dimensions in _AVATAR_SIZE.iteritems():
            avatars._size_to_url[dimensions] = urljoin(base_url, name)

        return avatars

    def as_dict(self):
        return dict(self._size_to_url)


mapper.add_mapping(YandexProxy.code, YandexProxy)
