# -*- coding: utf-8 -*-

from datetime import datetime
import inspect
import logging

from passport.backend.social.common.chrono import (
    datetime_to_unixtime,
    now,
)
from passport.backend.social.common.exception import (
    AlbumNotExistsProxylibError,
    BasicProxylibError,
    DatabaseError,
    InvalidTokenProxylibError,
    NetworkProxylibError,
    UnrefreshableTokenError,
)
from passport.backend.social.common.misc import (
    GraphiteMessageType,
    write_graph_log_message,
)

from . import get_proxy
from .refresh_token import refresh_token_by_token_id


# Появление этих исключений означает, что токен действительный.
EXCEPTIONS_FOR_VALID_TOKEN = (AlbumNotExistsProxylibError,)

logger = logging.getLogger()


def filter_function_kwargs(func, kwargs):
    """
    Создаёт новый словарь из данного, такой что, если ключ есть в kwargs и ключ
    совпадает с именем параметра функции func, то он есть в новом словаре.
    """
    # let's pass some params to function, but only params function expects to receive!
    if hasattr(func, 'real_co_varnames'):
        # для задекорированных функций
        varnames = func.real_co_varnames
    elif inspect.getargspec(func).keywords is not None:
        # для незадекорированных функций, умеющих принимать **kwargs
        varnames = kwargs.keys()
    else:
        varnames = func.func_code.co_varnames
    return {k: kwargs[k] for k in kwargs if k in varnames}


def tokens_to_proxies(tokens):
    """
    По списку токенов создаёт список с проксями, которые будут ходить
    использовать эти токены.
    """
    proxies = []
    for token in tokens:
        proxies.append(
            get_proxy(
                code=None if not token.application.provider else token.application.provider['code'],
                access_token=token.to_dict_for_proxy(),
                app=token.application,
            ),
        )
    return proxies


class MultitokenProxyMethodCaller(object):
    def __init__(self, tokens, refresh_tokens_enabled=True):
        self.proxies = tokens_to_proxies(tokens)
        self.last_called_proxy = None
        self._refresh_tokens_enabled = refresh_tokens_enabled
        self._tokens_for_refresh = []

        if refresh_tokens_enabled:
            for proxy in self.proxies:
                if self._proxy_uses_expired_token(proxy):
                    self._tokens_for_refresh.append(proxy.r.access_token['value'])

    def _proxy_uses_expired_token(self, proxy):
        token = proxy.r.access_token.get('value')
        expired_at = proxy.r.access_token.get('expires')
        if expired_at is not None:
            expired_at = datetime.fromtimestamp(expired_at)
        return token is not None and expired_at is not None and expired_at < now()

    def _refresh_token_in_proxy(self, proxy):
        if 'token_id' not in proxy.r.access_token:
            return
        try:
            access_token, expired = refresh_token_by_token_id(proxy.r.access_token['token_id'])

            self._tokens_for_refresh.remove(proxy.r.access_token.get('value'))

            proxy.r.access_token['value'] = access_token
            proxy.r.access_token['expires'] = datetime_to_unixtime(expired)
        except UnrefreshableTokenError:
            self._tokens_for_refresh.remove(proxy.r.access_token.get('value'))
            proxy = None
        return proxy

    def call_proxy_method(self, method_name, overall_kwargs):
        """
        Вызывает метод прокси, используя все известные токены, до первой
        успешной попытки.

        Подновляет токены с помощью Refresh-токенов.
        """
        unused_proxies = list(self.proxies)

        token_is_valid = False
        fails = []
        while unused_proxies and not token_is_valid:
            proxy = self.last_called_proxy = unused_proxies.pop(0)

            if self._refresh_tokens_enabled and proxy.r.access_token.get('value') in self._tokens_for_refresh:
                try:
                    proxy = self._refresh_token_in_proxy(proxy)
                except NetworkProxylibError as e:
                    fails.append(e)
                    logger.warning('Failed to refresh token because of network error')
                    write_graph_log_message(GraphiteMessageType.error, 'refresh_token', proxy.code, 'network')
                    proxy = None
                except DatabaseError as e:
                    fails.append(e)
                    logger.warning('Failed to refresh token because of database error')
                    proxy = None

            if proxy is not None:
                try:
                    proxy_method = getattr(proxy, method_name)
                    wanted_kwargs = filter_function_kwargs(proxy_method, overall_kwargs)
                    retval = proxy_method(**wanted_kwargs)
                    token_is_valid = True
                except BasicProxylibError as e:
                    fails.append(e)
                    if isinstance(e, EXCEPTIONS_FOR_VALID_TOKEN):
                        token_is_valid = True

        if not token_is_valid:
            # Если ни одна попытка не увенчалась успехом и произошла хотя бы одна
            # временная ошибка (сетевая, мигнула БД), то лучше сообщить, что следует
            # попробовать ещё раз.
            for fail in fails:
                if isinstance(fail, (DatabaseError, NetworkProxylibError)):
                    raise fail
            if fails:
                raise fails[-1]
            raise InvalidTokenProxylibError()

        elif fails and isinstance(fails[-1], EXCEPTIONS_FOR_VALID_TOKEN):
            raise fails[-1]

        return retval
