# -*- coding: utf-8 -*-

from datetime import timedelta
import logging
import operator

from passport.backend.social.common.chrono import now
from passport.backend.social.common.db.utils import (
    get_master_engine,
    get_slave_engine,
)
from passport.backend.social.common.exception import (
    DatabaseError,
    InvalidTokenProxylibError,
    NotFound,
    UnrefreshableTokenError,
)
from passport.backend.social.common.refresh_token.utils import (
    filter_refreshable_tokens,
    find_refresh_token_by_token_id,
    save_refresh_token,
)
from passport.backend.social.common.token.utils import (
    find_token_by_token_id,
    save_token,
)

from . import get_proxy


logger = logging.getLogger()


def refresh_token_by_token(access_token):
    """
    Исключения
        UnrefreshableTokenError
        NetworkProxylibError
        ProviderTemporaryUnavailableProxylibError
        UnexpectedResponseProxylibError
    """
    if not access_token:
        raise UnrefreshableTokenError()

    proxy = get_proxy(app=access_token.application)

    if not hasattr(proxy, 'refresh_token'):
        raise UnrefreshableTokenError()

    refresh_token = find_refresh_token_by_token_id(access_token.token_id, get_slave_engine())
    if refresh_token is None:
        raise UnrefreshableTokenError()

    try:
        response = proxy.refresh_token(refresh_token.value)
    except InvalidTokenProxylibError:
        raise UnrefreshableTokenError()

    access_token.value = response['access_token']
    access_token.expired = now() + timedelta(seconds=response['expires_in'])

    logger.debug('Token refreshed')

    try:
        save_token(access_token, get_master_engine())
        if 'refresh_token' in response:
            refresh_token.value = response['refresh_token']
            save_refresh_token(refresh_token, get_master_engine())
        logger.debug('Refreshed token saved: token_id=%s' % access_token.token_id)
    except DatabaseError:
        logger.warning('Failed to save refreshed token: token_id=%s' % access_token.token_id)


def refresh_token_by_token_id(token_id):
    """
    Исключения
        UnrefreshableTokenError
        NetworkProxylibError
        ProviderTemporaryUnavailableProxylibError
        UnexpectedResponseProxylibError
    """
    access_token = find_token_by_token_id(token_id, get_slave_engine())
    refresh_token_by_token(access_token)
    return (access_token.value, access_token.expired)


def filter_and_refresh_newest_token(tokens):
    """
    Находит среди данных токенов самый свежий действующий токен, а если он
    протух пробует его обновить

    # Исключения
        NetworkProxylibError
        NotFound
        ProviderTemporaryUnavailableProxylibError
        UnexpectedResponseProxylibError
    """
    tokens = sorted(tokens, key=operator.attrgetter('created'), reverse=True)

    not_expired_tokens = []
    expired_tokens = []
    for token in tokens:
        if token.is_going_to_expire():
            expired_tokens.append(token)
        else:
            not_expired_tokens.append(token)

    if not_expired_tokens:
        newest_token = not_expired_tokens[0]
    elif expired_tokens:
        refreshable_tokens = filter_refreshable_tokens(expired_tokens, get_slave_engine())
        if refreshable_tokens:
            newest_token = refreshable_tokens[0]
            try:
                refresh_token_by_token(newest_token)
            except UnrefreshableTokenError:
                raise NotFound('Failed to refresh token')
        else:
            raise NotFound('No refreshable tokens')
    else:
        raise NotFound('No tokens')

    return newest_token
