# -*- coding: utf-8 -*-

from __future__ import unicode_literals

import logging

from passport.backend.social.common.exception import UnexpectedResponseProxylibError
from passport.backend.social.proxylib.repo import mapping
from passport.backend.social.proxylib.repo.common import Repo


logger = logging.getLogger('proxylib.repo.mr')


class ProfileDoesNotHaveMoiMirAccountError(UnexpectedResponseProxylibError):
    pass


class MailRuRepo(Repo):
    code = 'mr'

    def parse_response(self):
        self.deserialize_json()
        self.parse_error_response()

    def convert_profile(self, profile):
        profile['userid'] = str(profile['userid'])
        if 'email' in profile:
            profile['username'] = profile['email']

    def convert_photo(self, photo):
        width = photo.pop('width')
        height = photo.pop('height')

        images = [
            {
                'url': photo.pop('src_small'),
                'width': min(width, 120),
                'height': min(height, 120),
            },
            {
                'url': photo.pop('src'),
                'width': min(width, 600),
                'height': min(height, 600),
            },
        ]
        if 'src_big' in photo:
            images.append({'url': photo.pop('src_big')})

        photo['images'] = images

    def extract_mails_unread_count(self):
        try:
            self.context['processed_data'] = self.context['data']['count']
        except (KeyError, TypeError):
            logger.warning('Failed to parse mails_unread_count response')
            self.context['processed_data'] = 0

    def parse_error_response(self):
        if not isinstance(self.context['data'], dict):
            return
        error = self.context['data'].get('error')
        if error is None:
            return
        self.raise_correct_exception(error['error_code'], error['error_msg'])

    def raise_correct_exception(self, error_code, error_message, error_description=None):
        if error_code == 204:
            # Если юзер удалил у себя Мой Мир, то АПИ вернёт нам ошибку 'empty_users'.
            logger.warning('The user doesn\'t have Moi Mir account. (code=%s, msg=%s)' % (error_code, error_message))
            raise ProfileDoesNotHaveMoiMirAccountError(error_message, error_description)
        else:
            super(MailRuRepo, self).raise_correct_exception(error_code, error_message, error_description)

    def detect_album_not_exists_error(self, error_code, error_message):
        # формат ответа при ошибке:
        # {"error":{"error_msg":"'aid' parameter is missing or invalid.","error_code":100}}
        return 'aid' in error_message


mapping[MailRuRepo.code] = MailRuRepo
