# -*- coding: utf-8 -*-

from __future__ import unicode_literals

import base64
import logging
import re

from passport.backend.core.types.phone_number.phone_number import InvalidPhoneNumber
from passport.backend.social.common.exception import (
    ProviderCommunicationProxylibError,
    UnexpectedResponseProxylibError,
)
from passport.backend.social.common.providers.Mts import Mts
from passport.backend.social.proxylib.repo import mapping
from passport.backend.social.proxylib.repo.common import (
    convert_phone_number,
    Repo,
)


logger = logging.getLogger('proxylib.repo.%s' % Mts.code)

_mts_error_message = re.compile(r'^com\.iplanet\.sso\.SSOException: (?P<error_message>.*)$')


class MtsRepo(Repo):
    code = Mts.code

    def parse_error_response(self):
        # Возможные сообщения об отказах не описаны, будем находить их в боевых
        # условиях.
        error_code = self.context['raw_response'].status
        if error_code // 100 != 2:
            response = self.context['raw_response'].decoded_data
            response_lines = response.splitlines()
            for response_line in response_lines[:2]:
                error_match = _mts_error_message.match(response_line)
                if error_match:
                    error_message = error_match.group('error_message')
                    break
            else:
                error_message = None
            self.raise_correct_exception(error_code, error_message)

    def convert_profile(self, profile):
        convert_userid(profile)
        profile.pop('personal_id', None)

        try:
            convert_phone_number(profile, 'RU')
        except InvalidPhoneNumber:
            raise UnexpectedResponseProxylibError()
        if not profile.get('phone'):
            raise UnexpectedResponseProxylibError()

        profile_type = profile.get('type') or ''
        if profile_type.lower() == 'organization':
            if 'organization_name' in profile:
                profile['firstname'] = profile['organization_name']
        else:
            if 'person_name' in profile:
                fullname = profile['person_name'].split(' ')
                profile['firstname'] = fullname[1]
                profile['lastname'] = fullname[0]

    def convert_profile_v3(self, profile):
        convert_userid_v3(profile)
        profile.pop('personal_id', None)

        try:
            convert_phone_number(profile, 'RU')
        except InvalidPhoneNumber:
            raise UnexpectedResponseProxylibError()
        if not profile.get('phone'):
            raise UnexpectedResponseProxylibError()

        if profile.get('is_organization') == 'Organization':
            if not profile.get('firstname') and profile.get('organization_name'):
                profile['firstname'] = profile['organization_name']
                profile.pop('lastname', None)
            profile.pop('organization_name', None)

        profile.pop('is_organization', None)

        self.convert_birthday(profile)


def convert_userid(profile):
    try:
        profile['userid'] = '%(phone)s:%(personal_id)s' % profile
    except KeyError:
        raise ProviderCommunicationProxylibError('Provider does not provide userid')


def convert_userid_v3(profile):
    # МТС пишет
    # > В кастомной реализации OAuth2 есть костыли специально для МТС Музыка,
    # > касающиеся особенностей работы этого приложения. В новой реализации подобных
    # > костылей быть не должно.
    # > Необходимо переделать новый функционал МТС Музыка. Для абонентов (используя
    # > наличие поля personalaccountnumber) применять старую логику. Для неабонентов
    # > (при отсутствии personalaccountnumber) - следует использовать значение
    # > Base64Encode(phone), например для 9625425070 => Base64Encode(9625425070) =>
    # > OTYyNTQyNTA3MA==, эту строку передавать Яндекс Музыка в качестве значения
    # > personalaccountnumber.
    if not profile.get('phone'):
        raise ProviderCommunicationProxylibError('Provider does not provide userid')

    personal_id = profile.get('personal_id')
    if not personal_id:
        personal_id = base64.standard_b64encode(profile.get('phone'))

    profile['userid'] = '%s:%s' % (profile.get('phone'), personal_id)


mapping[MtsRepo.code] = MtsRepo
