# -*- coding: utf-8 -*-

from __future__ import unicode_literals

import datetime
from functools import partial
import hashlib
import json
import logging
import re

from dateutil.parser import parse as parse_datetime_general
from jwt import decode as jwt_decode
from jwt.exceptions import DecodeError as JwtDecodeError
from lxml import etree
from passport.backend.core.types.phone_number.phone_number import InvalidPhoneNumber
from passport.backend.social.common.exception import (
    AlbumNotExistsProxylibError,
    CaptchaNeededProxylibError,
    InvalidTokenProxylibError,
    PermissionProxylibError,
    ProviderCommunicationProxylibError,
    ProviderRateLimitExceededProxylibError,
    ProviderTemporaryUnavailableProxylibError,
    ResponseDecodeProxylibError,
    SocialUserDisabledProxylibError,
    UnexpectedResponseProxylibError,
    ValidationRequiredProxylibError,
)
from passport.backend.social.common.misc import (
    copy,
    trim_message,
    urlencode,
)
from passport.backend.social.common.provider_settings import providers
from passport.backend.social.common.social_config import social_config
import passport.backend.social.proxylib.useragent
import phonenumbers
from werkzeug.urls import url_decode


logger = logging.getLogger('proxylib.repo.common')

EMPTY_SET = frozenset()


def convert_phone_number(profile, country):
    phone_number = str(profile.get('phone') or '')
    if phone_number:
        phone_number = phonenumbers.parse(phone_number, country)
        if not phonenumbers.is_valid_number(phone_number):
            raise InvalidPhoneNumber()
        profile['phone'] = phonenumbers.format_number(phone_number, phonenumbers.PhoneNumberFormat.E164)
    else:
        profile['phone'] = None


def parse_id_token(id_token):
    try:
        id_token = jwt_decode(id_token, verify=False)
    except JwtDecodeError:
        logger.debug('Failed to parse JWT: %s' % trim_message(id_token))
        raise UnexpectedResponseProxylibError('id_token is not JWT')
    return id_token


class Repo(object):
    xml_parser = etree.XMLParser(remove_blank_text=True, resolve_entities=False, no_network=True)

    def __init__(self, code):
        self.access_token = {}
        self.app = None
        self.aux_data = {}
        self.context = {}
        self.provider_code = code
        self.settings = {}
        self.should_retry_on_invalid_token = False

    def execute_request_basic(self, http_method=None, timeout=None, retries=None,
                              parser=None):
        url = self.context['request']['url']
        data = self.context['request'].get('data')
        if http_method is None:
            http_method = 'post' if data is not None else 'get'
        if social_config.broker_stress_mode:
            url = social_config.broker_stress_emulator_provider_url + '/'.join(url.split('/')[3:])  # pragma: no cover

        provider = providers.providers.get(self.provider_code)
        if retries is None and provider:
            retries = provider.get('retries')
        if timeout is None and provider:
            timeout = provider.get('timeout')

        service_temporary_unavailable_exceptions = [
            ProviderTemporaryUnavailableProxylibError,
            UnexpectedResponseProxylibError,
        ]
        if self.should_retry_on_invalid_token:
            # У провайдеров часто отстают реплики и они считают вновь
            # выданный токен недействительным. Поэтому мы делаем
            # повторную попытку, когда получаем такой отказ.
            service_temporary_unavailable_exceptions.append(InvalidTokenProxylibError)

        try:
            self.context['raw_response'] = passport.backend.social.proxylib.useragent.execute_request(
                method=http_method,
                url=url,
                fields=data,
                headers=self.context['request'].get('headers'),
                timeout=timeout,
                retries=retries,
                request_signer=self.get_request_signer(),
                from_intranet=self.app.request_from_intranet_allowed,
                parser=partial(self._parse_http_response, parser=parser),
                service_temporary_unavailable_exceptions=service_temporary_unavailable_exceptions,
                service=self._get_service_for_useragent(),
            )
        except ProviderTemporaryUnavailableProxylibError as e:
            service_temporary_unavailable_exception = copy(
                e.service_temporary_unavailable_exception,
                fallback=e.service_temporary_unavailable_exception,
            )
            service_temporary_unavailable_exception.cause = e
            raise service_temporary_unavailable_exception

    def _get_service_for_useragent(self):
        if self.provider_code:
            return self.provider_code
        if self.app:
            return self.app.name

    def get_request_signer(self):
        return

    def _parse_http_response(self, http_response, parser):
        self.context['raw_response'] = http_response
        if not parser:
            parser = self.parse_response
        parser()
        return self.context['data']

    def parse_response(self):
        self.context['data'] = self.context['raw_response']

    def extract_response_data(self, mapping, converter=None, one=False, listed=True, extract_field=None):
        items = []

        data_source = self.context['data']
        if isinstance(extract_field, basestring):
            extract_field = [extract_field]

        if isinstance(extract_field, (list, tuple)):
            for field in extract_field:
                data_source = data_source[field]

        def _process_item(input_item_local):
            if not isinstance(input_item_local, dict):
                raise ProviderCommunicationProxylibError('Provider returned unexpected data')
            if not input_item_local:
                raise ProviderCommunicationProxylibError('Provider returned nothing')
            item = {}
            for key, value in input_item_local.iteritems():
                if key in mapping:
                    new_key = mapping[key].split('.')
                    insert_point = item
                    while len(new_key) > 1:
                        if new_key[0] not in insert_point:
                            insert_point[new_key[0]] = {}
                        insert_point = insert_point[new_key[0]]
                        del new_key[0]
                    insert_point[new_key[0]] = value
            if converter is not None:
                converter(item)

            # удалим пустые элементы
            for key in item.keys():
                if item[key] is None:
                    del item[key]
            return item

        if listed:  # перед нами список элементов (возможно, всегда из одного элемента)
            for input_item in data_source:
                items.append(_process_item(input_item))
        else:  # перед нами одиночный элемент
            items = [_process_item(data_source)]

        self.context['processed_data'] = items[0] if one else items

    def compose_request(self, url_name=None, method=None, additional_args=None, data=None,
                        fields=None, base_url=None, additional_headers=None, add_access_token=True,
                        userid=None, add_application_key=None):
        token_name = self.settings.get('access_token_parameter_name', 'access_token')
        if not base_url:
            base_url = self.settings[url_name]
        headers = {}
        args = {}
        if additional_args:
            args.update(additional_args)
        if fields:
            args['fields'] = ','.join(fields)

        captcha_sid = self.aux_data.get('captcha_sid')
        captcha_answer = self.aux_data.get('captcha_answer')
        if captcha_sid and captcha_answer:
            args.update({
                'captcha_sid': captcha_sid,
                'captcha_key': captcha_answer,
            })

        # add application id if needed
        if self.settings.get('add_app_id'):
            key_name = self.settings.get('app_id_key_name') or 'app_id'
            args[key_name] = self.app.custom_provider_client_id or self.app.id

        # add application key if needed
        if add_application_key is None:
            add_application_key = self.settings.get('add_application_key')
        if add_application_key:
            key_name = self.settings.get('application_key_key_name') or 'application_key'
            args[key_name] = self.app.key

        # Add method type
        if method:
            new_url = base_url % {'method': method}
            if new_url == base_url:
                args['method'] = method
            else:
                base_url = new_url

        # TODO Унести подписывание запроса в RequestSigner и вызывать его из
        # useragent.
        # sign a request if needed
        if self.settings.get('should_sign'):
            signature_name = self.settings.get('signature_name', 'sig')

            if self.settings.get('sign_with_token') and add_access_token:
                args[token_name] = self.access_token['value']

            if self.settings.get('signature_type') == 'md5':
                sep = self.settings.get('signature_key_value_separator', '=')
                args[signature_name] = self.generate_md5_signature(args, key_value_separator=sep, userid=userid)
            else:
                args[signature_name] = self.generate_signature(args)

        # add access token (even if it has already been added)
        if add_access_token:
            if self.settings.get('oauth_auth_over_header'):
                headers = {'Authorization': 'Bearer ' + self.access_token['value']}
            else:
                args[token_name] = self.access_token['value']

        # add more headers if needed
        if additional_headers:
            headers.update(additional_headers)

        more_headers = self.settings.get('additional_headers')
        if more_headers:
            headers.update(more_headers)

        # turn empty dict into None (I'm not crazy as of your first thought)
        headers = headers if headers else None

        if self.app and self.app.app_server_key:
            args['app_server_key'] = self.app.app_server_key

        self.context['request'] = dict(
            url=base_url + ('?' + urlencode(args) if args else ''),
            data=data,
            headers=headers,
        )

    def compose_profile_url_oauth1(self, url_name, method, additional_args=None,
                                   http_method='GET', additional_post_data=None):
        url = self.settings[url_name] % {'method': method}
        if additional_args:
            url += '?' + urlencode(additional_args)
        self.context['request'] = dict(url=url, headers=dict(), data=additional_post_data)

    def deserialize_json(self):
        raw_response = self.context['raw_response']
        try:
            response = raw_response.decoded_data
        except UnicodeDecodeError:
            logger.debug('Response is not valid UTF8: %s' % trim_message(raw_response.content))
            raise ResponseDecodeProxylibError('Response is not valid UTF8')

        try:
            self.context['data'] = json.loads(response)
        except ValueError:
            logger.debug('Failed to parse JSON: %s' % trim_message(response))
            raise ResponseDecodeProxylibError('Response is not JSON')

    def deserialize_jwt(self):
        response = self.context['raw_response'].decoded_data
        try:
            self.context['data'] = jwt_decode(response, verify=False)
        except JwtDecodeError:
            logger.debug('Failed to parse JWT: %s' % trim_message(response))
            raise UnexpectedResponseProxylibError('Response is not JWT')

    def deserialize_urlencoded(self):
        # Не выбрасывает исключений, даже если на входе каша
        self.context['data'] = url_decode(self.context['raw_response'].decoded_data)

    def deserialize_xml(self):
        response = self.context['raw_response'].data
        try:
            self.context['data'] = etree.XML(response, parser=self.xml_parser)
        except etree.LxmlError:
            logger.debug('Failed to parse XML: %s' % trim_message(response))
            raise UnexpectedResponseProxylibError('Response is not XML')

    def convert_extract_field(self, field_name):
        self.context['data'] = self.context['data'][field_name]

    def convert_birthday(self, source=None):
        if source is None:
            source = self.context['processed_data']
        regexp = self.settings['birthday_regexp']
        bdate = source.get('birthday')
        if bdate is None:
            return

        res = re.match(regexp, bdate)
        if not res:
            del source['birthday']
            return

        items = {}
        values = res.groupdict()
        for key in ['year', 'month', 'day']:
            items[key] = int(values[key]) if values.get(key) else None

        bday = None
        if 'month' in items and 'day' in items:
            bday = '%s-%s-%s' % (items['year'] if 'year' in items and items['year'] else '0000',
                                 unicode(items['month']).zfill(2),
                                 unicode(items['day']).zfill(2))
        if bday is not None:
            source['birthday'] = bday

    def convert_friends_birthday(self):
        for user in self.context['processed_data']:
            self.convert_birthday(source=user)

    def convert_photo_albums_privacy(self, mapping):
        for album in self.context['processed_data']:
            priv = album.get('visibility')
            if priv is None:
                album['visibility'] = 'private'
                continue
            if not isinstance(priv, list):
                priv = [priv]
            priv = map(unicode, priv)
            priv = [mapping.get(p.lower(), 'private') for p in priv]
            if all(p == 'public' for p in priv):
                album['visibility'] = 'public'
            elif all(p in {'friends', 'public'} for p in priv):
                album['visibility'] = 'friends'
            else:
                album['visibility'] = 'private'

    def convert_gender(self, source=None):
        gmap = self.settings['gender_map']
        if source is None:
            source = self.context['processed_data']

        gender = source.get('gender')
        if gender is None:
            return
        if gender not in gmap:
            del source['gender']
        else:
            source['gender'] = gmap[gender]

    def convert_friends_gender(self):
        for user in self.context['processed_data']:
            self.convert_gender(source=user)

    def should_remove_avatar(self, key, url):
        if not url:
            return True
        regexp = self.settings.get('avatars_exclude_regexp')
        if not regexp:
            return False
        return bool(regexp.match(url))

    def filter_stub_avatars(self, source=None):
        if source is None:
            source = self.context['processed_data']
        avatar_dict = source.get('avatar', {})
        for key in avatar_dict.keys():
            url = avatar_dict[key]
            if self.should_remove_avatar(key, url):
                del avatar_dict[key]

    def generate_md5_signature(self, params, key_value_separator='=', userid=None):
        logger.debug('Generating signature. Params are: %s' % params)

        params_str = ''.join('%s%s%s' % (k, key_value_separator, unicode(v)) for k, v in sorted(params.items()))
        if userid:
            plain_signature = '%s%s%s' % (userid, params_str, self.app.key)
        else:
            plain_signature = '%s%s' % (params_str, self.app.secret)

        md5_signature = hashlib.md5(plain_signature.encode('utf-8')).hexdigest()
        logger.debug('Signature basestring is %s' % plain_signature)
        logger.debug('MD5 signature is %s' % md5_signature)
        return md5_signature

    def generate_signature(self, params):
        logger.debug('PARAMS %s' % sorted(params.items()))

        signature_1 = ''.join(u'%s=%s' % (k, unicode(v)) for k, v in sorted(params.items()))
        signature_2 = '%s%s' % (self.access_token['value'], self.app.secret)
        logger.debug('PLAIN SIGNATURE 1 %s' % signature_1)
        logger.debug('PLAIN SIGNATURE 2 %s' % signature_2)
        sig_2 = hashlib.md5(signature_2.encode('utf-8')).hexdigest()
        sig = hashlib.md5(signature_1.encode('utf-8') + sig_2).hexdigest()
        return sig

    def detect_album_not_exists_error(self, error_code, error_message):
        """
        Соц сети при указании несуществующего альбома обычно возвращают малоинформативный код,
        одзначающий только "неверный или неуказанный параметр" и что-то еще. Вот из этого "еще"
        можно пытаться определить, правда ли это ошибка отсутствующего альбома.
        """
        return True

    def raise_correct_exception(self, error_code, error_message, error_description=None):
        error_message = trim_message(error_message, cut=False)
        logger.debug('Detecting error type for error=%s...' % error_code)

        # Ошибка отсутствия альбома у пользователя очень сложно детектируется, поэтому здесь много магии.
        photo_methods = ['get_photos', 'get_user_photos', 'photo_post_get_request', 'photo_post_commit']
        if (
            getattr(self, 'method_name', None) in photo_methods and
            error_code in self.settings.get('album_not_exists', EMPTY_SET) and
            self.detect_album_not_exists_error(error_code, error_message)
        ):
            logger.info('Album existence detected: ' + unicode(error_message))
            raise AlbumNotExistsProxylibError(error_message)

        if error_code in self.settings.get('error_codes_permission', EMPTY_SET):
            logger.info('Permission error: ' + unicode(error_message))
            raise PermissionProxylibError(error_message)
        elif error_code in self.settings.get('error_codes_invalid_token', EMPTY_SET):
            logger.info('Invalid token: ' + unicode(error_message))
            raise InvalidTokenProxylibError(error_message)
        elif error_code in self.settings.get('error_rate_limit_exceeded', EMPTY_SET):
            logger.info('Rate limit exceeded: ' + unicode(error_message))
            raise ProviderRateLimitExceededProxylibError(error_message)
        elif error_code in self.settings.get('error_user_blocked', EMPTY_SET):
            logger.info('User error detected: ' + unicode(error_message))
            raise SocialUserDisabledProxylibError(error_message)
        elif error_code in self.settings.get('error_captcha_needed', EMPTY_SET):
            sid = self.context['data']['error']['captcha_sid']
            url = self.context['data']['error']['captcha_img']
            logger.info('User has to enter captcha. sid=%s', sid)
            raise CaptchaNeededProxylibError(sid=sid, url=url)
        elif error_code in self.settings.get('error_validation_required', EMPTY_SET):
            validation_url = self.context['data']['error']['redirect_uri']
            logger.info('User has to pass validation at %s' % validation_url)
            raise ValidationRequiredProxylibError(validation_url=validation_url)
        elif error_code in self.settings.get('error_codes_service_unavailable', EMPTY_SET):
            raise ProviderTemporaryUnavailableProxylibError(error_message)
        else:
            response = self.context.get('raw_response')
            response = '' if not response else response.decoded_data
            logger.error('Unexpected provider error: code=%s, msg=%s, response=%s' % (error_code, error_message, response))
            raise UnexpectedResponseProxylibError(error_message, error_description)

    def parse_datetime(self, item, field):
        if field not in item or not item[field]:
            return
        try:
            # Тупой, но работающий способ получить timestamp
            dt = parse_datetime_general(str(item[field]))
            diff = (dt - datetime.datetime(1970, 1, 1, tzinfo=dt.tzinfo))
            timestamp = diff.days * 86400 + diff.seconds
            item[field] = timestamp
        except ValueError:
            return

    @staticmethod
    def parse_display_name(display_name):
        items = display_name.strip().split(' ', 1)
        firstname = items[0]
        lastname = items[1] if len(items) == 2 else None
        return firstname, lastname
