# -*- coding: utf-8 -*-

from datetime import datetime
from functools import partial
import json
import logging

from passport.backend.core.types.ip.ip import IP
from passport.backend.social.common.chrono import now
from passport.backend.social.common.context import request_ctx
from passport.backend.social.common.db.execute import execute
from passport.backend.social.common.db.schemas import (
    profile_table,
    token_table,
)
from passport.backend.social.common.db.utils import (
    get_master_engine,
    get_slave_engine,
)
from passport.backend.social.common.exception import (
    ApplicationUnknown,
    CaptchaNeededProxylibError,
    DatabaseError,
    InvalidTokenProxylibError,
    NetworkProxylibError,
    OfflineZoraUseragentNotImplementedError,
    ProviderCommunicationProxylibError,
    ProviderRateLimitExceededProxylibError,
    ProviderUnknownProxylibError,
    SocialUserDisabledProxylibError,
    UnrefreshableTokenError,
    ValidationRequiredProxylibError,
)
from passport.backend.social.common.misc import split_scope_string
from passport.backend.social.common.provider_settings import providers
from passport.backend.social.common.redis_client import get_redis
from passport.backend.social.common.social_config import social_config
from passport.backend.social.common.token.domain import Token
from passport.backend.social.common.token.utils import (
    delete_token_by_token_id,
    save_token,
)
from passport.backend.social.proxylib import get_proxy
from passport.backend.social.proxylib.refresh_token import refresh_token_by_token
from passport.backend.social.utils.zookeeper import run_exclusively
from sqlalchemy import sql


logger = logging.getLogger(__name__)


def dump_token(token):
    return json.dumps(token.to_json_dict())


def delete_token(token, reason, write_engine):
    logger.info('Deleting token with id=%s' % token.token_id)
    logger.info('TOKEN=%s' % dump_token(token))
    deleted_rows = delete_token_by_token_id(token.token_id, write_engine)
    if deleted_rows:
        logger.info('[app_id=%s][%s]Token deleted' % (token.application_id, reason))
    else:
        logger.warning('Token was not deleted')


def update_token(token, write_engine):
    logger.debug('Updating token %s' % token.token_id)
    save_token(token, write_engine)
    logger.info('Token updated: %s' % token.token_id)


def get_profile_by_id(profile_id, read_engine):
    pt = profile_table
    query = sql.select([pt]).where(pt.c.profile_id == profile_id)
    profile = execute(read_engine, query).fetchone()
    return profile


def process_token(token, read_engine, write_engine):
    token = Token(
        uid=token.uid,
        token_id=token.token_id,
        profile_id=token.profile_id,
        application_id=token.application_id,
        value=token.value,
        secret=token.secret,
        scopes=split_scope_string(token.scope),
        expired=token.expired,
        created=token.created,
        verified=token.verified,
        confirmed=token.confirmed,
    )

    logger.info('Processing token %d' % token.token_id)

    app = providers.get_application_by_id(token.application_id)
    if not app:
        delete_token(token, 'bad_application', write_engine)
        return
    request_ctx.application = token.application = app

    if token.expired and token.expired < now():
        logger.debug('Token is outdated due to database "expired" field')
        try:
            refresh_token_by_token(token)
        except UnrefreshableTokenError:
            delete_token(token, 'db_expired', write_engine)
            return

    if token.profile_id is None:
        return

    logger.info('Checking whether corresponding profile exists...')
    profile = get_profile_by_id(token.profile_id, read_engine)
    if not profile:
        delete_token(token, 'missing_profile', write_engine)
        return

    provider_info = providers.get_provider_info_by_id(profile.provider_id)
    if not provider_info:
        delete_token(token, 'bad_provider', write_engine)
        return
    code = provider_info['code']
    logger.debug('Provider: %s' % code)

    proxy = get_proxy(code, access_token=token.to_dict_for_proxy(), app=app, aux_data=dict(user_ip=IP('127.0.0.1')))

    if code == 'fb':
        try:
            token_info = proxy.get_token_info()
        except InvalidTokenProxylibError:
            delete_token(token, 'invalid', write_engine)
            return

        if not token_info['valid']:
            delete_token(token, 'invalid', write_engine)
            return

        is_updated = False

        # check scopes
        if sorted(token.scopes) != sorted(token_info['scopes']):
            token.scopes = token_info['scopes']
            is_updated = True

        # check expiration time
        true_expiration_unixtime = datetime.fromtimestamp(token_info['expires'])
        if true_expiration_unixtime != token.expired:
            token.expired = true_expiration_unixtime
            is_updated = True

        if is_updated:
            update_token(token, write_engine)
    else:
        try:
            proxy.get_profile()
        except InvalidTokenProxylibError:
            delete_token(token, 'invalid', write_engine)
            return
        except SocialUserDisabledProxylibError:
            delete_token(token, 'social_user_disabled', write_engine)
            return


def clean_db_zk(tokens_per_transaction):
    # Токены удаляются под Zookeeper'ом, из-за того что нынешняя схема
    # удаления, не защищена от параллельного удаления одних и тех же токенов из
    # разных воркеров.
    run_exclusively(
        partial(clean_db, tokens_per_transaction),
        lock_name='/socialism/clean_tokens.lock',
        zookeeper_hosts=social_config.zookeeper_hosts,
    )


def clean_db(tokens_per_transaction):
    logger.info('Scanning...')
    last_token_id = get_redis().get('dbscripts_last_token_id') or 0
    logger.debug('last_token_id=%s' % last_token_id)

    tt = token_table
    query = sql.select([tt]).where(tt.c.token_id > last_token_id).limit(tokens_per_transaction).order_by(tt.c.token_id)
    tokens = execute(get_slave_engine(), query).fetchall()

    new_last_token_id = tokens[-1].token_id if len(tokens) == tokens_per_transaction else 0
    logger.debug('Setting last_token_id=%s' % new_last_token_id)
    get_redis().set('dbscripts_last_token_id', new_last_token_id)

    for token in tokens:
        try:
            process_token(token, get_slave_engine(), get_master_engine())
        except NetworkProxylibError:
            logger.warning('Network failed')
        except DatabaseError:
            logger.warning('Database failed')
        except ProviderCommunicationProxylibError as e:
            logger.error('Top level exception: %s (message=%s, description=%s)' % (type(e).__name__, e.message, e.description))
        except ProviderUnknownProxylibError as e:
            logger.error('Top level exception: %s (%s)' % (type(e).__name__, e))
        except (
            ApplicationUnknown,
            CaptchaNeededProxylibError,
            ProviderRateLimitExceededProxylibError,
            ValidationRequiredProxylibError,
        ) as e:
            logger.error('Top level exception: %s' % type(e).__name__)
        except OfflineZoraUseragentNotImplementedError as e:
            logger.error('Failed to process token because offline Zora useragent not implemented')
        except Exception:
            logger.error('Top level exception', exc_info=True)
        finally:
            request_ctx.clear()
