# -*- coding: utf-8 -*-

from __future__ import unicode_literals

import datetime
import logging
import logging.config
import shlex
import sys
import time

import celery.app
from passport.backend.core.lazy_loader import LazyLoader
from passport.backend.social.common.importer import (
    install_file_system_importer,
    install_python_path2,
)
from passport.backend.social.common.chrono import now
from passport.backend.social.common.db.utils import (
    get_master_engine,
    get_slave_engine,
)
from passport.backend.social.common.social_config import social_config
from passport.backend.social.common.token.utils import (
    find_token_by_token_id,
    save_token,
)
from passport.backend.social.utils import logging_settings
import retry


# Признак, что все токены обработаны
JOBS = dict()

logger = logging.getLogger(__name__)


def main():
    logging.captureWarnings(True)
    logging.config.dictConfig(logging_settings.LOGGING)

    install_python_path2()
    install_file_system_importer()

    social_config.init()

    LazyLoader.register('chrono', lambda: datetime.datetime)
    LazyLoader.get_instance('chrono')

    import social_configs.celeryconfig
    celery.app.app_or_default().config_from_object(social_configs.celeryconfig)

    # Читаем идентификатор задачи
    if len(sys.argv) < 2:
        logger.error('Job not specified')
        return 1
    job = JOBS.get(sys.argv[1])
    if not callable(job):
        logger.error('Job not found: %s' % sys.argv[1])
        return 1

    args = parse_job_kwargs(sys.argv[2:])
    job(**args)

    return 0


def parse_job_kwargs(args):
    return {args[2 * i]: args[2 * i + 1] for i in range(len(args) / 2)}


def job(f):
    def wrapper(**kwargs):
        import passport.backend.social.utils.tasks

        batch = kwargs.pop('batch', None)
        retries = int(kwargs.pop('retries', 10))

        rps = kwargs.pop('rps', None)
        rps = int(rps) if rps is not None else rps

        task = getattr(passport.backend.social.utils.tasks, f.__name__)
        if not batch:
            task.delay(**kwargs)
            return

        schedule_task = task.delay
        if rps is not None:
            schedule_task = Throttled(rps, task.delay)

        lines = sys.stdin if batch == '-' else open(batch, 'r')
        for line in lines:
            line_kwargs = dict(kwargs, **parse_job_kwargs(shlex.split(line.strip())))
            retry.retry_call(schedule_task, None, line_kwargs, tries=retries)

    JOBS[f.__name__] = wrapper
    return f


@job
def eval_token_hash(token_id):
    logger.debug('eval_token_hash:work on %s' % token_id)

    token = find_token_by_token_id(int(token_id), get_slave_engine())

    if token is not None:
        save_token(token, get_master_engine())


class Throttled(object):
    def __init__(self, rps, f):
        self.delay = 0
        self.f = f
        self.max_round_size = 2
        self.max_rps = rps
        self.round_size = 0
        self.round_started_at = None
        self.total_max_round_size = rps

    def __call__(self, *args, **kwargs):
        if self.round_started_at is None:
            self.round_started_at = now.f()

        if self.round_size >= self.max_round_size:
            round_time = now.f() - self.round_started_at - self.max_round_size * self.delay
            round_rps = self.max_round_size / float(round_time)
            if round_rps > self.max_rps:
                self.delay = 1. / self.max_rps - 1. / round_rps
                self.max_round_size *= 2
                if self.max_round_size > self.total_max_round_size:
                    self.max_round_size = self.total_max_round_size
            else:
                self.delay = 0
                self.max_round_size = 2

            self.round_started_at = now.f()
            self.round_size = 0

        time.sleep(self.delay)
        self.round_size += 1

        return self.f(*args, **kwargs)


if __name__ == '__main__':
    main()
