# -*- coding: utf-8 -*-

import logging
from logging.config import dictConfig as logging_dict_config

import celery.signals
from celery.task import task
from passport.backend.social.common.context import request_ctx
from passport.backend.social.common.gunicorn import setup_worker
from passport.backend.social.common.misc import build_request_id

from . import (
    logging_settings,
    one_time_job,
)
from .dbcleaner import clean_db_zk as _clean_db
from .init import init


logger = logging.getLogger('passport.backend.social.utils.tasks')


@task
def ping():
    logger.info('pong')


@task
def clean_db(tokens_per_transaction):
    _clean_db(tokens_per_transaction)


@celery.signals.worker_ready.connect
def on_worker_ready(*arg, **kwargs):
    setup_worker()
    init()


@task(bind=True, acks_late=True)
def eval_token_hash(task, token_id):
    try:
        one_time_job.eval_token_hash(token_id)
    except Exception:
        logger.error('eval_token_hash:Unhandled exception, token_id=%s' % token_id, exc_info=True)
        task.retry()


@celery.signals.task_prerun.connect
def on_task_prerun(*arg, **kwargs):
    request_ctx.request_id = build_request_id()


@celery.signals.task_postrun.connect
def on_task_postrun(*arg, **kwargs):
    request_ctx.clear()


@celery.signals.after_setup_logger.connect
def celery_after_setup_logger(*args, **kwargs):
    logging.captureWarnings(True)
    logging_dict_config(logging_settings.LOGGING)
