# -*- coding: utf-8 -*-
from collections.abc import Iterable
from functools import wraps

from flask import request
from passport.backend.core.grants import (
    get_grants,
    GrantsError,
    MissingGrantsError,
    MissingTicketError,
    TicketParsingError,
)
from passport.backend.core.logging_utils.loggers.statbox import StatboxLogger
from passport.backend.takeout.api.base import (
    exception_to_response,
    json_response,
    render_form_error,
)
from passport.backend.takeout.common.statbox import exception_to_statbox
from six import string_types
from werkzeug.datastructures import MultiDict
import yenv


def get_request_values_func():
    if request.method in ['POST', 'PUT']:
        consumer = request.args.get('consumer', '')
        values = MultiDict(request.form)
        values.update({'consumer': consumer})
        values.update(request.files)
        return values
    else:
        return MultiDict(request.args)


def validate(form_cls, error_renderer=None):
    error_renderer = error_renderer or render_form_error

    def decorator(f):
        @wraps(f)
        def wrapper(*args, **kwargs):
            values = get_request_values_func()
            values.update(kwargs)
            # new_args dict провалидированные параметры через форму
            form = form_cls(formdata=values)
            if form.validate():
                new_args = form.data
                return f(new_args)
            else:
                return error_renderer(form)

        return wrapper

    return decorator


def grants(required_grants):
    def decorator(f):
        @wraps(f)
        def wrapper(form_values, *args, **kwargs):
            consumer = form_values['consumer']
            try:
                get_grants().check_access(
                    request.env.consumer_ip,
                    [consumer],
                    required_grants,
                    service_ticket=request.env.service_ticket,
                )
            except MissingGrantsError as e:
                return json_response(
                    403,
                    status='error',
                    error=u'Access denied for ip: %s; consumer: %s; tvm_client_id: %s. Required grants: %s' %
                          (e.ip, e.consumer, e.tvm_client_id, e.missing),
                    env=yenv.type,
                )
            except TicketParsingError as e:
                return json_response(
                    403,
                    status='error',
                    error=u'Access denied for ip: %s; consumer: %s; tvm_client_id: %s; Error: %s; Debug info: %s' %
                          (e.ip, e.consumer, e.tvm_client_id, e.message, e.debug_info),
                    env=yenv.type,
                )
            except MissingTicketError as e:
                # TODO на эту ветку кода надо написать тест, когда паспорт заедет в аркадию
                # core/test/grants.py:FakeGrants.set_grant_list сейчас не позволяет
                # переопределить client_id
                return json_response(
                    403,
                    status='error',
                    error=u'Access denied for ip: %s; consumer: %s; tvm_client_id: %s; Error: Missing TVM ticket' %
                          (e.ip, e.consumer, e.tvm_client_id),
                    env=yenv.type,
                )
            except GrantsError as e:
                return json_response(
                    403,
                    status='error',
                    error=u'Access denied for ip: %s; consumer: %s; tvm_client_id: %s' %
                          (e.ip, e.consumer, e.tvm_client_id),
                    env=yenv.type,
                )

            try:
                return f(form_values, *args, **kwargs)
            except Exception as e:
                return exception_to_response(e)

        return wrapper

    return decorator


def statbox_oneline_logger(logger_cls=StatboxLogger, base_params=None, exclude_form_fields=None):
    base_params = base_params or {}
    exclude_form_fields = exclude_form_fields or []

    def decorator(f):
        @wraps(f)
        def wrapper(form_values, *args, **kwargs):
            form_fields_to_log = [field for field in sorted(form_values.keys()) if field not in exclude_form_fields]

            for form_field_name in form_fields_to_log:
                form_value = form_values[form_field_name]
                if isinstance(form_value, Iterable) and not isinstance(form_value, string_types):
                    base_params[form_field_name] = ', '.join(map(str, form_value))
                else:
                    base_params[form_field_name] = form_value

            base_params.update(
                request_id=request.env.request_id,
                request_path=request.path,  # урл, который дёрнули
                ip=request.env.consumer_ip,
                log_source='api',
                tskv_format='takeout-log',  # NOTE это под вопросом
            )

            logger = logger_cls(**base_params)

            with exception_to_statbox(logger):
                result = f(form_values, *args, statbox=logger, **kwargs)
            logger.log()
            return result

        return wrapper

    return decorator
