# -*- coding: utf-8 -*-
import logging
import os

from flask import (
    has_request_context,
    request,
)
from passport.backend.core.logging_utils.helpers import (
    escape_text_for_log,
    trim_message,
)
from passport.backend.core.logging_utils.request_id import RequestIdManager
from passport.backend.utils.string import smart_text
from werkzeug.datastructures import FileStorage


try:
    from flask.wrappers import JSONMixin
except ImportError:
    # JSONMixin and Response merged in Flask 2
    from flask.wrappers import Response as JSONMixin


# в nginx request_id это 32 символа
REQUEST_ID_LAST_CHARS_COUNT = 32


log = logging.getLogger('takeout.api.common')


def get_param_value(param_name):
    if has_request_context():
        value = request.values.get(param_name)
    else:
        value = None
    return value or '-'


def get_external_request_id():
    if has_request_context():
        request_id = request.env.request_id
        if request_id:
            return request_id[-REQUEST_ID_LAST_CHARS_COUNT:]
    return '-'


def setup_log_prefix():
    clear_log_prefix()

    request_id = (
        os.getpid(),
        get_external_request_id(),
    )
    RequestIdManager.push_request_id(*request_id)

    request_id = (
        get_param_value('uid'),
    )

    RequestIdManager.push_request_id(*request_id)


def clear_log_prefix():
    RequestIdManager.clear_request_id()


def format_parameters_for_log(params, trim=False):
    keys_and_values = []
    for key, value in params.items(multi=True):
        if isinstance(value, FileStorage):
            value = value.filename
        key_and_value = '{}={}'.format(
            escape_text_for_log(key, errors='replace'),
            escape_text_for_log(value, errors='replace'),
        )
        keys_and_values.append(key_and_value)
    message = ', '.join(keys_and_values)
    if trim:
        message = trim_message(message)
    return smart_text(message, errors='replace')


def log_request():
    # Строчка в логе для упрощения грепа. Урл и параметры в одной строке.
    # request_id добавляется форматтером, поэтому здесь не упоминается
    log.info(u'Request: ' + u'\t'.join([
        request.method + ' ' + request.path,
        # GET не обрезаем, т.к. он уже ограничен by design длиной урла
        u'args: ' + format_parameters_for_log(request.args, trim=False),
        # POST обрезаем от греха подальше
        u'form: ' + format_parameters_for_log(request.form, trim=True),
        # файлы
        u'files: ' + format_parameters_for_log(request.files, trim=True),
    ]))

    if request.env.request_id:
        log.info(u'Request id: %s', request.env.request_id)


def log_response(response):
    response_data = response.data
    response_data = trim_message(response_data.decode('utf-8'))

    log.info(u'Response status: %s', response.status)

    if isinstance(response, JSONMixin) and response.is_json:
        keys = {u'status', u'error'}
        response_dict = response.get_json()
        essential_fields = {key: str(response_dict[key]) for key in keys if response_dict.get(key) is not None}
        essential_data = u', '.join(u'%s=%s' % (k, v) for k, v in essential_fields.items())
        log.info(u'Response sent: %s; Full: "%s"', essential_data, response_data)
    else:
        log.info(u'Response sent: "%s"', response_data)

    clear_log_prefix()

    return response
