# -*- coding: utf-8 -*-
from passport.backend.core.logging_utils.request_id import RequestIdManager
from passport.backend.core.protobuf.takeout.takeout_pb2 import TakeoutRequest
from passport.backend.takeout.api.base import (
    json_response,
    ok_response,
)
from passport.backend.takeout.api.decorators import (
    grants,
    statbox_oneline_logger,
    validate,
)
from passport.backend.takeout.api.grants import GRANT_DEBUG_INTEGRATION_TEST
from passport.backend.takeout.api.views.forms import DebugIntegrationForm
from passport.backend.takeout.api.views.prepare_archive import get_extract_id
from passport.backend.takeout.common.conf.services import get_service_configs
from passport.backend.takeout.common.logbroker import get_takeout_logbroker_writer
from passport.backend.utils.time import get_unixtime
import yenv


@validate(DebugIntegrationForm)
@statbox_oneline_logger(
    base_params={
        'action': 'debug_integration_test',
    },
)
@grants([GRANT_DEBUG_INTEGRATION_TEST])
def run_integration_test(args, statbox):
    uid = args['uid']
    unixtime = args['unixtime'] or get_unixtime()
    extract_id = get_extract_id()
    consumer = args['consumer']
    service_name = args['service_name']

    statbox.bind(
        extract_id=extract_id,
        archive_requested_unixtime=unixtime,
    )

    # Добавление в логи информации об extract_id и service_name
    RequestIdManager.push_request_id(extract_id, service_name)

    service_configs = get_service_configs()
    if service_name not in service_configs:
        error_message = 'Unknown service: {}'.format(repr(service_name))

        statbox.bind(
            error='unknown_service',
            error_message=error_message,
        )

        return json_response(
            status_code=400,
            status='error',
            error=error_message,
        )

    if consumer not in service_configs[service_name].run_integration_test_consumers:
        error_message = 'Consumer {} cannot run integration test for service {}'.format(
            repr(consumer),
            repr(service_name),
        )

        statbox.bind(
            error='grants_error',
            error_message=error_message,
        )

        return json_response(
            status_code=403,
            status='error',
            error=error_message,
            env=yenv.type,
        )

    task_base_message = TakeoutRequest.TaskBaseMessage(
        uid=uid,
        extract_id=extract_id,
        unixtime=unixtime,
        task_name='extract_service',
    )
    task_detail_message = TakeoutRequest.TaskExtractServiceMessage(
        service=service_name,
        step='start',
    )

    logbroker = get_takeout_logbroker_writer('takeout_tasks')

    logbroker.send(
        TakeoutRequest(task_base_message=task_base_message, extract_service=task_detail_message)
    )

    statbox.bind(
        status='ok',
        service_name=service_name,
        step='start',
    )

    return ok_response(
        uid=uid,
        extract_id=extract_id,
        service_name=service_name,
        env=yenv.type,
    )
