# -*- coding: utf-8 -*-
from flask_wtf.file import (
    FileField,
    FileRequired,
)
from passport.backend.takeout.common.exceptions import (
    BadJobId,
    BadSignature,
)
from passport.backend.takeout.common.job_id import parse_job_id
from passport.backend.takeout.common.utils import is_filename_valid_for_archive
from wtforms import Form
from wtforms.fields import (
    BooleanField,
    IntegerField,
    SelectMultipleField,
    StringField,
)
from wtforms.validators import (
    InputRequired,
    NumberRange,
    Optional,
    ValidationError,
)


MAX_UID = 2 ** 63 - 2  # Все что больше ЧЯ не считает валидным UID-ом


class FormWithConsumer(Form):
    consumer = StringField(
        label='consumer',
        validators=[
            InputRequired(),
        ],
    )


class MultipleField(SelectMultipleField):
    """
    Поле формы для доступа ко всем значениям, переданным в форму под определённым именем
    """
    def pre_validate(self, form):
        pass


def valid_job_id(form, field):
    try:
        job_id = field.data
        parse_job_id(job_id)
    except (BadJobId, BadSignature) as e:
        raise ValidationError(e.__class__.__name__)


def _check_filename_is_valid(filename):
    if not is_filename_valid_for_archive(filename):
        raise ValidationError('Invalid filename {}'.format(filename))


def validate_fileobj_filename(form, field):
    file_obj = field.data
    if file_obj is None:
        return
    _check_filename_is_valid(file_obj.filename)


def validate_filenames(form, field):
    for filename in field.data or []:
        _check_filename_is_valid(filename)


class DebugIntegrationForm(FormWithConsumer):
    uid = IntegerField(
        label='uid',
        validators=[
            InputRequired(),
            NumberRange(0, MAX_UID, message='UID is out of range'),
        ],
    )
    service_name = StringField(
        label='service_name',
        validators=[
            InputRequired(),
        ],
    )
    unixtime = IntegerField(
        label='unixtime',
        validators=[
            Optional(),
            NumberRange(0, message='unixtime is out of range'),
        ],
    )


class DebugMakeArchive(FormWithConsumer):
    uid = IntegerField(
        label='uid',
        validators=[
            InputRequired(),
            NumberRange(0, MAX_UID, message='UID is out of range'),
        ],
    )
    extract_id = StringField(validators=[InputRequired()])


class PrepareArchiveForm(FormWithConsumer):
    uid = IntegerField(
        label='uid',
        validators=[
            InputRequired(),
            NumberRange(0, MAX_UID, message='UID is out of range'),
        ],
    )
    unixtime = IntegerField(
        label='unixtime',
        validators=[
            InputRequired(),
            NumberRange(0, message='unixtime is out of range'),
        ],
    )
    manual_takeout = BooleanField(
        label='manual_takeout',
        validators=[Optional()],
    )


class UploadForm(FormWithConsumer):
    job_id = StringField(
        label='job_id',
        validators=[
            InputRequired(),
            valid_job_id,
        ],
    )
    file = FileField(
        label='file',
        validators=[
            FileRequired(),
            validate_fileobj_filename,
        ],
    )


class DoneForm(FormWithConsumer):
    job_id = StringField(
        label='job_id',
        validators=[
            InputRequired(),
            valid_job_id,
        ],
    )
    filename = MultipleField(
        label='filename',
        validators=[
            validate_filenames,
        ]
    )


class GetDebugJobIdForm(FormWithConsumer):
    uid = IntegerField(
        label='uid',
        validators=[
            InputRequired(),
            NumberRange(0, MAX_UID, message='UID is out of range'),
        ],
    )
    service_name = StringField(validators=[InputRequired()])
    extract_id = StringField(validators=[InputRequired()])


class DebugStartProcessForm(FormWithConsumer):
    uid = IntegerField(
        label='uid',
        validators=[
            InputRequired(),
            NumberRange(0, MAX_UID, message='UID is out of range'),
        ],
    )
    service_name = StringField(validators=[InputRequired()])
    extract_id = StringField(validators=[InputRequired()])


class DebugGetStatusForm(FormWithConsumer):
    uid = IntegerField(
        label='uid',
        validators=[
            InputRequired(),
            NumberRange(0, MAX_UID, message='UID is out of range'),
        ],
    )
    extract_id = StringField(validators=[InputRequired()])
    service_name = MultipleField()


class FakeServiceFileForm(Form):
    file_name = StringField(label='file_name', default='example.txt')


class FakeServiceAsyncUploadStartForm(Form):
    uid = IntegerField(
        label='uid',
        validators=[
            InputRequired(),
            NumberRange(0, MAX_UID, message='UID is out of range'),
        ],
    )
    unixtime = StringField(validators=[InputRequired()])
    job_id = StringField(validators=[InputRequired()])


class PingForm(Form):
    check = StringField(label='check', default='')
