# -*- coding: utf-8 -*-
import logging

from flask import jsonify
from passport.backend.core.logging_utils.request_id import RequestIdManager
from passport.backend.takeout.api.base import (
    ok_response,
    render_form_error as orig_render_form_error,
)
from passport.backend.takeout.api.decorators import (
    grants,
    statbox_oneline_logger,
    validate,
)
from passport.backend.takeout.api.grants import (
    GRANT_DEBUG_MAKE_ASYNC_UPLOAD_JOB_ID,
    GRANT_DEBUG_START_ASYNC_UPLOAD_PROCESS,
    GRANT_UPLOAD_DONE,
    GRANT_UPLOAD_FILE,
)
from passport.backend.takeout.api.views.forms import (
    DebugStartProcessForm,
    DoneForm,
    GetDebugJobIdForm,
    UploadForm,
)
from passport.backend.takeout.common.conf import get_config
from passport.backend.takeout.common.job_id import (
    make_job_id_v1,
    parse_job_id,
)
from passport.backend.takeout.common.touch import TouchFiles
from passport.backend.takeout.common.utils import (
    encrypt_and_upload_to_s3,
    s3_get_uploaded_files,
)


upload_log = logging.getLogger('takeout.api.proxy.upload')


def render_form_error(form):
    if 'job_id' in form.errors:
        rv = jsonify(
            status='job_id.invalid',
            error=', '.join(form.errors['job_id'])
        )
        rv.status_code = 400
        return rv
    return orig_render_form_error(form)


@validate(DebugStartProcessForm)
@statbox_oneline_logger(
    base_params={
        'action': 'debug_start_process',
    },
)
@grants([GRANT_DEBUG_START_ASYNC_UPLOAD_PROCESS])
def debug_start_process(args, statbox):
    uid = args['uid']
    service_name = args['service_name']
    extract_id = args['extract_id']

    RequestIdManager.push_request_id(extract_id, service_name)

    touch = TouchFiles(uid, extract_id, service_name)
    touch.set(TouchFiles.AsyncUpload.STARTED)
    touch.set(TouchFiles.AsyncUpload.ORDERED_DATA)

    job_id = make_job_id_v1(uid, service_name, extract_id)

    statbox.bind(
        job_id=job_id,
        status='ok',
    )

    return ok_response(
        job_id=job_id,
    )


@validate(GetDebugJobIdForm)
@statbox_oneline_logger(
    base_params={
        'action': 'debug_get_job_id',
    },
)
@grants([GRANT_DEBUG_MAKE_ASYNC_UPLOAD_JOB_ID])
def get_debug_job_id(args, statbox):
    uid = args['uid']
    service_name = args['service_name']
    extract_id = args['extract_id']

    RequestIdManager.push_request_id(extract_id, service_name)

    job_id = make_job_id_v1(uid, service_name, extract_id)

    statbox.bind(
        job_id=job_id,
        status='ok',
    )

    return ok_response(
        job_id=job_id,
    )


@validate(UploadForm, error_renderer=render_form_error)
@statbox_oneline_logger(
    base_params={
        'action': 'async_upload',
    },
    exclude_form_fields=['file'],
)
@grants([GRANT_UPLOAD_FILE])
def upload(args, statbox):
    job_id = parse_job_id(args['job_id'])
    file_obj = args['file']

    RequestIdManager.push_request_id(job_id.extract_id, job_id.service_name)

    statbox.bind(
        uid=job_id.uid,
        extract_id=job_id.extract_id,
        service_name=job_id.service_name,
        filename=file_obj.filename,
    )

    if job_id.service_name not in get_config()['services']:
        # FIXME Подобные места - хорошие претенденты на рефакторинг кода
        # Чтобы ошибочные ответы не отдавать во вьюхах, а кидать исключения
        # и в самом конце стека уже исключения транслировать в ошибки api
        error_message = 'Unknown service {}'.format(job_id.service_name)

        statbox.bind(
            error='unknown_service',
            error_message=error_message,
        )

        return ok_response(
            status='job_id.invalid',
            error=error_message,
        )

    touch = TouchFiles.from_job_id(job_id)

    # не готовы к заливке
    # был пропущен первый шаг интеграции?
    if not touch.is_set(TouchFiles.AsyncUpload.STARTED):
        error_message = 'Not ready to receive files'

        statbox.bind(
            error='process_not_started',
            error_message=error_message,
        )

        return ok_response(
            status='job_id.invalid',
            error=error_message,
        )

    if touch.is_set(TouchFiles.AsyncUpload.DONE):
        error_message = 'Task is complete'

        statbox.bind(
            error='process_already_done',
            error_message=error_message,
        )

        return ok_response(
            status='job_id.invalid',
            error=error_message,
        )

    encrypt_and_upload_to_s3(
        uid=job_id.uid,
        extract_id=job_id.extract_id,
        service_name=job_id.service_name,
        file_name=file_obj.filename,
        file_object=file_obj,
    )

    statbox.bind(status='ok')

    return ok_response()


@validate(DoneForm, error_renderer=render_form_error)
@statbox_oneline_logger(
    base_params={
        'action': 'async_upload_done',
    },
)
@grants([GRANT_UPLOAD_DONE])
def upload_done(args, statbox):
    job_id = parse_job_id(args['job_id'])
    should_be_uploaded_filenames = set(args.get('filename', []))

    statbox.bind(
        uid=job_id.uid,
        extract_id=job_id.extract_id,
        service_name=job_id.service_name,
    )

    RequestIdManager.push_request_id(job_id.extract_id, job_id.service_name)

    if job_id.service_name not in get_config()['services']:
        error_message = 'Unknown service {}'.format(job_id.service_name)

        statbox.bind(
            error='unknown_service',
            error_message=error_message,
        )

        return ok_response(
            status='job_id.invalid',
            error=error_message,
        )

    touch = TouchFiles.from_job_id(job_id)

    if touch.is_set(TouchFiles.AsyncUpload.DONE):
        statbox.bind(status='ok')
        return ok_response()

    # не готовы к заливке
    # был пропущен первый шаг интеграции?
    if not touch.is_set(TouchFiles.AsyncUpload.STARTED):
        error_message = 'Not ready to receive files'

        statbox.bind(
            error='process_not_started',
            error_message=error_message,
        )

        return ok_response(
            status='job_id.invalid',
            error=error_message,
        )

    uploaded_filenames = set(s3_get_uploaded_files(
        job_id.uid,
        job_id.extract_id,
        job_id.service_name,
    ))

    if uploaded_filenames != should_be_uploaded_filenames:
        extra_files = sorted(uploaded_filenames - should_be_uploaded_filenames)
        missing_files = sorted(should_be_uploaded_filenames - uploaded_filenames)
        response_data = {}
        if extra_files:
            response_data['extra_files'] = extra_files
        if missing_files:
            response_data['missing_files'] = missing_files

        statbox.bind(
            error='files_mismatch',
            error_message='Disprepancy between uploaded files and stated files',
            extra_files=', '.join(extra_files),
            missing_files=', '.join(missing_files),
        )
        return ok_response(
            status='missing',
            **response_data
        )

    touch.set(TouchFiles.AsyncUpload.DONE)

    statbox.bind(status='ok')

    return ok_response()
