# -*- coding: utf-8 -*-
from __future__ import absolute_import

import logging
import re
from urllib.parse import urljoin

from passport.backend.core.builders.base.base import (
    BaseBuilder,
    RequestInfo,
)
from passport.backend.core.builders.mixins.json_parser.json_parser import JsonParserMixin
from passport.backend.core.logging_utils.helpers import trim_message
from passport.backend.core.logging_utils.loggers import GraphiteLogger
from passport.backend.takeout.common.conf import get_config
from passport.backend.takeout.common.conf.services import get_service_configs
from passport.backend.takeout.common.constants import (
    STATUS_ERROR,
    STATUS_NO_DATA,
    STATUS_OK,
    STATUS_PENDING,
)
from passport.backend.takeout.common.exceptions import (
    BaseServiceError,
    ResponseNotReadyError,
    ServicePermanentError,
    ServiceTemporaryError,
)
from passport.backend.takeout.common.utils import is_filename_valid_for_archive


log = logging.getLogger('takeout.builders')


FILE_NAME_RE = re.compile(r'filename=([^;]+)')

DEFAULT_TIMEOUT = 5
DEFAULT_RETRIES = 3


def error_handler(response):
    if response.status_code != 200:
        log.warning(
            u'Request failed with with response=%s code=%s',
            trim_message(response.content.decode('utf-8')),
            response.status_code,
        )
        raise ServiceTemporaryError('Got http_status=%s' % response.status_code)


def common_error_detector(response, raw_response):
    log.debug('Got status %r from service', response.get('status'))
    if response.get('status') == STATUS_ERROR or response.get('error'):
        raise ServiceTemporaryError('Got error=%r' % response.get('error', ''))


def raw_file_error_detector(response, raw_response):
    # Здесь не логирую status, потому что его нет. Сюда приходит пара file_name, file_object
    pass


def sync_get_error_detector(response, raw_response):
    common_error_detector(response, raw_response)
    if response.get('status') not in (STATUS_OK, STATUS_NO_DATA):
        raise ServicePermanentError('Invalid status: %s' % response.get('status'))
    if response['status'] == STATUS_OK and ('data' not in response and 'file_links' not in response):
        raise ServicePermanentError('Response has no data or file_links but status is OK')


def async_get_error_detector(response, raw_response):
    common_error_detector(response, raw_response)
    if response.get('status') not in (STATUS_OK, STATUS_PENDING, STATUS_NO_DATA):
        raise ServicePermanentError('Invalid status: %s' % response.get('status'))
    if response.get('status') == STATUS_PENDING:
        raise ResponseNotReadyError('Not all services are ready')
    if response['status'] == STATUS_OK and ('data' not in response and 'file_links' not in response):
        raise ServicePermanentError('Response has no data or file_links but status is OK')


def async_start_error_detector(response, raw_response):
    common_error_detector(response, raw_response)
    if response.get('status') not in (STATUS_OK, STATUS_NO_DATA):
        raise ServicePermanentError('Invalid status: %s' % response.get('status'))
    if response.get('status') != STATUS_NO_DATA:
        if 'job_id' not in response:
            raise ServicePermanentError('No job_id in response')
        if not response['job_id']:
            raise ServicePermanentError('Got empty job_id')


def async_upload_start_error_detector(response, raw_response):
    common_error_detector(response, raw_response)
    if response.get('status') not in (STATUS_OK, STATUS_NO_DATA):
        raise ServicePermanentError('Invalid status: %s' % response.get('status'))


class BaseServiceBuilder(BaseBuilder, JsonParserMixin):
    """
    Базовый класс билдера для внешних сервисов. В параметре conf принимает все настройки проекта.
    """

    base_error_class = BaseServiceError
    temporary_error_class = ServiceTemporaryError
    parser_error_class = ServicePermanentError

    expected_service_type = None

    def __init__(self, service_name, conf=None):
        conf = conf or get_config()
        base_config = conf['builders']
        service_config = self._get_service_config(service_name, conf=conf)
        super(BaseServiceBuilder, self).__init__(
            url=service_config.url_base,
            timeout=service_config.timeout or base_config['default_timeout'],
            retries=service_config.retries or base_config['default_retries'],
            logger=log,
            graphite_logger=GraphiteLogger(service=service_name),
            tvm_dst_alias=service_config.tvm_dst_alias,
        )
        self.url_suffix_start = service_config.url_suffix_start
        self.url_suffix_get = service_config.url_suffix_get

    @classmethod
    def _get_service_config(cls, service_name, conf=None):
        available_configs = get_service_configs(conf=conf)
        if service_name not in available_configs:
            raise ValueError('Unknown service: %s' % service_name)
        config = available_configs[service_name]
        if config.type != cls.expected_service_type:
            raise ValueError(
                'Invalid builder type selected for `%s`: %s != %s' % (
                    service_name,
                    cls.expected_service_type,
                    config.type,
                ),
            )
        return config

    def get_info_from_url(self, file_link):
        raise NotImplementedError('Must be overriden in subclasses, if supported')

    def parse_raw_file(self, raw_response):
        content_disposition_header = raw_response.headers.get('content-disposition')
        if not content_disposition_header:
            raise self.parser_error_class('No Content-Disposition header')

        search_result = FILE_NAME_RE.search(content_disposition_header)
        if not search_result:
            raise self.parser_error_class('Malformed Content-Disposition header: %s' % content_disposition_header)

        file_name = search_result.group(1).strip('"')

        if not is_filename_valid_for_archive(file_name):
            raise self.parser_error_class('Got unacceptable filename {}'.format(repr(file_name)))

        file_object = raw_response.raw
        file_object.decode_content = True
        return file_name, file_object

    def _make_request(self, url_suffix=None, custom_url=None, method='POST', post_data=None, stream=False,
                      error_detector=None, parser=None):
        if not custom_url and url_suffix is None:
            raise ValueError('Either `custom_url` or `url_suffix` is required')  # noqa
        return self._request_with_retries(
            method=method,
            request_info=RequestInfo(
                url=custom_url or urljoin(self.url, url_suffix),
                get_args=None,
                post_args=post_data,
            ),
            http_error_handler=error_handler,
            error_detector=error_detector or common_error_detector,
            parser=parser or self.parse_json,
            stream=stream,
        )


class SyncServiceBuilder(BaseServiceBuilder):
    expected_service_type = 'sync'

    def get_info(self, uid,  unixtime):
        return self._make_request(
            post_data=dict(
                uid=uid,
                unixtime=unixtime,
            ),
            url_suffix=self.url_suffix_get,
            error_detector=sync_get_error_detector,
        )

    def get_info_from_url(self, custom_url):
        return self._make_request(
            custom_url=custom_url,
            method='GET',
            stream=True,
            parser=self.parse_raw_file,
            error_detector=raw_file_error_detector,
        )


class AsyncServiceBuilder(BaseServiceBuilder):
    expected_service_type = 'async'

    def start(self, uid,  unixtime):
        return self._make_request(
            post_data=dict(
                uid=uid,
                unixtime=unixtime,
            ),
            url_suffix=self.url_suffix_start,
            error_detector=async_start_error_detector,
        )

    def get_info(self, job_id):
        return self._make_request(
            post_data=dict(
                job_id=job_id,
            ),
            url_suffix=self.url_suffix_get,
            error_detector=async_get_error_detector,
        )

    def get_info_from_url(self, custom_url):
        return self._make_request(
            custom_url=custom_url,
            method='GET',
            stream=True,
            parser=self.parse_raw_file,
            error_detector=raw_file_error_detector,
        )


class AsyncUploadServiceBuilder(BaseServiceBuilder):
    expected_service_type = 'async_upload'

    def start(self, uid,  unixtime, job_id):
        return self._make_request(
            post_data=dict(
                uid=uid,
                unixtime=unixtime,
                job_id=job_id,
            ),
            url_suffix=self.url_suffix_start,
            error_detector=async_upload_start_error_detector,
        )
